/***********************************************************************************
 *                      PragmaDev RTDS win32 integration
 ***********************************************************************************
 *                             SOCKET COMMUNICATION FUNCTIONS
 *----------------------------------------------------------------------------------
 * The purpose of these functions is to connect to the host environment through
 * a socket.
 * The information sent through the socket is used
 *     - from the target to the host: to trace SDL events
 *     - from host to target: to stop the target program and send messages to the
 *         system
 * ---------------------------------------------------------------------------------
 *                    + RTDS_InitSocket
 *                    + RTDS_SendSocket
 *                    + RTDS_ReadSocket
 *                    + RTDS_CloseSocket
 ***********************************************************************************/

#ifdef RTDS_SOCKET_PORT

/* INCLUDES*/
/* Std include */
#include <stdlib.h>
#include <stdio.h>

#ifdef __CYGWIN__
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#else
#include <Winsock2.h>
#endif

/* RTDS kernel includes */
#include "RTDS_OS.h"
#include "RTDS_MACRO.h"
#include "RTDS_Error.h"

/* TCP Client include*/
#include "RTDS_TCP_Client.h"

/***********************************************************************************************************
 *                                 FUNCTION  RTDS_InitSocket(...)
 *----------------------------------------------------------------------------------------------------------
 *    Author    : Pragmadev S.P
 *    Date        : 06/06/02
 *    Purpose : start socket
 ***********************************************************************************************************/
RTDS_SOCKET_ID_TYPE RTDS_InitSocket( char * hostNameString, int portNum )
    {
    int sd;
    int rc = -1;
    struct sockaddr_in localAddr;
    struct sockaddr_in servAddr;
    int clientSocketId = 0;
    
#ifndef __CYGWIN__
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;
    wVersionRequested = MAKEWORD( 2, 2 );
    err = WSAStartup( wVersionRequested, &wsaData );
#endif
    
    servAddr.sin_family = AF_INET;
    servAddr.sin_addr.s_addr = inet_addr( hostNameString );  /* Server IP address */
    servAddr.sin_port = htons( portNum );
    
    /* Create socket */
    sd = socket( AF_INET, SOCK_STREAM, 0 );
    
    if( sd < 0 )
        {
        /* Cannot open socket */
        return RTDS_ERROR;
        }
    else
        {
        clientSocketId = sd;
        }
    /* Bind any port number */
    localAddr.sin_family = AF_INET;
    localAddr.sin_addr.s_addr = htonl( INADDR_ANY );
    localAddr.sin_port = htons( 0 );
    
    rc = bind( sd, ( struct sockaddr * ) &localAddr, sizeof( localAddr ) );
    
    if( rc < 0 )
        {
        /* Cannot bind port */
        return RTDS_ERROR;
        }
    
    rc = connect( sd, ( struct sockaddr * ) &servAddr, sizeof( servAddr ) );
    
    if( rc != 0 )
        {
        /* Could not connect to server */
        return RTDS_ERROR;
        }
    return clientSocketId;
    }

/******************************************************************************************
 *                                 FUNCTION  RTDS_CloseSocket(...)
 *-----------------------------------------------------------------------------------------
 *    Purpose  :
 *
 *    return status RTDS_OK / RTDS_ERROR
 *
 ******************************************************************************************/
int RTDS_CloseSocket( RTDS_SOCKET_ID_TYPE socketId )
    {
    /* Since no way to stop task if not self : this function    doesn't attempt to kill thread */
#ifdef __CYGWIN__
    close( socketId );
#else
    closesocket( socketId );
#endif
    
    return RTDS_OK;
    }

/******************************************************************************************
 *                                 FUNCTION  RTDS_ReadSocket(...)
 *-------------------------------------------------------------------
 *    Purpose : return a valid data (ended by END_STRING) or RTDS_ERROR
 *                        if socket connection and reading probleme.
 *
 *    Release : This version of function read and process char by char
 *                and does not use any buffer to read a certain
 *                        amount of data before processing.
 *
 *    socketDescriptor : socket ID
 *    data                         : read buffer: pointer on the value to return
 *    delimiterChar      : delimiter char
 *
 *
 *    return                     : currentSize or -1 if failing
 ******************************************************************************************/
int RTDS_ReadSocket( RTDS_SOCKET_ID_TYPE socketId, char ** data, int * dataSize, char delimiterChar, char escapeChar )
    {
    static int n;
    int offset = 0;
    char * tmpReadBuffer;
    int blockSize = 1024;
    int blockFreeSpace;     /* Used to extend memory chunk if needed */
    int nbBlock = 1;        /* Used to count the number of block allocated for the reda buffer */
    char tmpchar;
    int escapeFlag = 0;     /* indicate that the previous received char was the escapeChar*/
    
    /* Init the block free space to the initial size */
    blockFreeSpace = blockSize;
    tmpReadBuffer = ( char * )RTDS_MALLOC( blockSize );
    *data = NULL;
    
    while( 1 )
        {
        n = recv( socketId, &tmpchar, sizeof( char ), 0 );
        if( n < 0 )
            {
            /* Cannot receive data */
            return -1;
            }
        else if( n == 0 )
            {
            /* Connection closed by server */
            RTDS_CloseSocket( socketId );
            return -1;
            }
        else if( n == 1 )
            {
            if( escapeFlag == 1 )
                {
                if ( tmpchar == delimiterChar )
                    {
                    /* escapeChar was followed by delimiterChar */
                    *dataSize = offset + 2;
                    tmpReadBuffer[ offset ] = tmpchar;
                    *data = tmpReadBuffer;
                    break;
                    }
                else
                    {
                    /* escapeChar was not followed by delimiterChar: wait for escapeChar */
                    escapeFlag = 0;
                    }
                }
                else
                    {
                    /* waiting for escapeChar */
                    if ( tmpchar == escapeChar )
                        {
                        escapeFlag = 1;
                        }
                    }
            /* Add char and increment offset */
            tmpReadBuffer[ offset ] = tmpchar;
            offset++;
            
            if ( !blockFreeSpace )
                {
                nbBlock++;
                tmpReadBuffer = (char *)realloc(tmpReadBuffer, nbBlock * blockSize);
                if ( tmpReadBuffer = NULL )
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_EXTEND_MEMORY_ALLOC );
                    }
                blockFreeSpace = blockSize;
                }
            else
                {
                blockFreeSpace--;
                }
            }/* end of else if n==1 */
        } /* end of while */
    
    return offset;
    }

/******************************************************************************************
 *                                 FUNCTION  RTDS_SendSocket(...)
 *-------------------------------------------------------------------
 *    Purpose : Send data over the socket
 *
 *    socketId                : socket ID
 *    commandString     : pointer on data to send
 *    size                        : size to send
 *
 *    return                    : RTDS_OK or RTDS_ERROR
 ******************************************************************************************/
int RTDS_SendSocket( RTDS_SOCKET_ID_TYPE socketId, char * commandString, int size )
    {
    int stat = -1;
    
    /* send msg */
    stat = send( socketId, commandString, ( size_t )size, 0 );
    
    if ( stat < 1 )
        {
        stat = RTDS_ERROR;
        }
    else
        {
        stat = RTDS_OK;
        }
    return stat;
    }

#endif /* RTDS_SOCKET_PORT */


