/* **************************************************************** *
 *              PragmaDev RTDS win32 integration 
 * **************************************************************** */

/* RTDS include */
#include "RTDS_MACRO.h"

/* Includes to use socket */
#include <winsock2.h> /* Replaces <sys/types.h> for 64 bits support */

#ifdef RTDS_SOCKET_PORT
#include "RTDS_TCP_Client.h"
#endif

#ifdef RTDS_FORMAT_TRACE
#include "RTDS_FormatTrace.h"
#endif

/*
 * SEMAPHORE PROTOTYPE:
 * -------------------
 * Variable declared as global to the all the program
 */
RTDS_SYS_TIME_PROTO;
RTDS_CRITICAL_SECTION_PROTO;
RTDS_START_SYNCHRO_PROTO;
RTDS_CRITICAL_TRACE_SECTION_PROTO;
RTDS_SOCKET_ACCESS_PROTO;
RTDS_DTRACE_ACKNOWLEDGE_PROTO;

#if defined( RTDS_SOCKET_PORT )
    extern int globalClientSocketId;
#endif

#ifdef RTDS_MEMORY_COUNTER
  #ifdef RTDS_PRINTF_AVAILABLE
    #include "stdio.h"
  #endif
  int RTDS_globalMallocCounter=0;
  
  void *RTDS_Malloc(int size)
    {
    void *ptr;
    RTDS_globalMallocCounter++;
    ptr = calloc( 1, size );
    #ifdef RTDS_PRINTF_AVAILABLE
    printf("RTDS_Malloc: ptr=%x, size=%d, RTDS_globalCounter=%d\n", ptr, size, RTDS_globalMallocCounter);
    #endif
    return ptr;
    }
    
  void RTDS_Free(void *ptr)
    {
    RTDS_globalMallocCounter--;
    #ifdef RTDS_PRINTF_AVAILABLE
    printf("RTDS_Free: ptr=%x, RTDS_globalCounter=%d\n", ptr, RTDS_globalMallocCounter);
    #endif
    free( ptr );
    }
#endif
  
/********************************************************************
 *                      RTDS_MsgQueueCreate
 *-------------------------------------------------------------------
 * The purpose of this function is to allocate memory for both
 * semaphore:
 * Chained_List_Sem (that protects Chained list) and Blocking_Sem
 * (that allows thread to wait for messages when message box is empty)
 *
 * This structure contains pointers to semaphore and pointer to
 * message box (chained list)
 *********************************************************************/
RTDS_QCB * RTDS_MsgQueueCreate()
    {
    RTDS_QCB * Queue_Control_Block;

    /* Memory allocation of the RTDS_QCB (Queue Control Block) */
    Queue_Control_Block = ( RTDS_QCB * )RTDS_MALLOC( sizeof( RTDS_QCB ) );
    
    /* Initialization of the Chained_listSem */
    /* Create semaphore only for this process and with initial value: unlocked */
    Queue_Control_Block->chainedListSemId = RTDS_SemMCreate();
    
    /* Initialization of the Blocking_Sem */
    Queue_Control_Block->blockingSemId = RTDS_SemCCreate( RTDS_SEMAPHORE_MAXCOUNT, 0 );
    
    /* Initialize Queue_Control_Block->queue to NULL */
    Queue_Control_Block->queue = NULL;
    
    return Queue_Control_Block;
    }

/********************************************************************
 *                    RTDS_MsgQueueSend
 *-------------------------------------------------------------------
 * Allocate memory for the new message to send and insert it in
 * the receiver Message Queue
 * (change the "next" pointer of the last message to point
 * to the new message)
 * **************************************************************** *
 * Parameters:
 *  - receiver              receiver queue id of the message
 *  - RTDS_messageToSend    message to send
 * Returns:
 *     Nothing
 ********************************************************************/
int RTDS_MsgQueueSend( RTDS_QCB * receiver, RTDS_MessageHeader * RTDS_messageToSend )
    {
    RTDS_MessageHeader * dynamicMessageToSend;
    RTDS_MessageHeader * msg = NULL;
    
    if ( RTDS_messageToSend != NULL )
        {
        dynamicMessageToSend = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
        memcpy( dynamicMessageToSend, RTDS_messageToSend, sizeof( RTDS_MessageHeader ) );

        /* Protect Chained list */
        RTDS_SemTake( receiver->chainedListSemId, RTDS_SEMAPHORE_TIME_OUT_FOREVER );

        /* initialize msg to Point on the first message */
        msg = receiver->queue;

        if ( msg != NULL )
            {
            /* Set msg to point on the last message */
            while( msg->next != NULL )
                {
                msg = msg->next;
                }
            msg->next = dynamicMessageToSend;
            /* Set the last message's "next" pointer to the adress of the message to send */
            }
        else
            {
            receiver->queue = dynamicMessageToSend;
            }

        /* Give Chained list semaphore */
        RTDS_SemGive( receiver->chainedListSemId );

        /* Give Blocking semaphore to allow thread to read message */
        RTDS_SemGive( receiver->blockingSemId );
        }

    return RTDS_OK;
    }

/********************************************************************
 *                     RTDS_MsgQueueDelete
 *-------------------------------------------------------------------
 * The purpose of this function is to delete a message queue:
 *     - delete all messages contained in the queue
 *     - delete the queue (and related semaphore)
 *********************************************************************/
int RTDS_MsgQueueDelete( RTDS_RtosQueueId queueId )
    {
    RTDS_MessageHeader * msg;
    RTDS_MessageHeader * nextMsg;
    
    /* 1- DELETE ALL MESSAGES */
    msg = queueId->queue;
    while( msg != NULL )
        {
        nextMsg = msg->next;
        RTDS_FREE( msg );
        msg = nextMsg;
        }
    /* 2- DELETE QUEUE CONTROL BLOCK */
    RTDS_SemDelete( queueId->chainedListSemId );
    RTDS_SemDelete( queueId->blockingSemId );
    
    RTDS_FREE( queueId );
    return RTDS_OK;
    }

/********************************************************************
 *                     FUNCTION  RTDS_SemMCreate
 *-------------------------------------------------------------------
 * Create a mutex semaphore
 *
 * PARAMETERS
 *        None
 * RETURN
 *        SemaphoreId : RTDS_SemaphoreId
 *********************************************************************/
RTDS_SemaphoreId RTDS_SemMCreate( void )
    {
    RTDS_SemaphoreId SemId;
    
    SemId = ( RTDS_SemaphoreId )RTDS_MALLOC( sizeof( RTDS_SemaphoreInfo ) );
    SemId->semId = CreateMutex( NULL, FALSE, NULL );
    if ( SemId->semId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_MUTEX );
        }
    else
        {
        SemId->semType = MUTEX;
        return SemId;
        }
    }

/********************************************************************
 *                  FUNCTION  RTDS_SemCCreate
 *-------------------------------------------------------------------
 * Create a counting semaphore
 *-------------------------------------------------------------------
 * PARAMETERS
 *        maxCount     : int         maximum semaphoe counter value
 *        initialState : int         initial semaphore counter value 
 * RETURN
 *        SemaphoreId : RTDS_SemaphoreId
 *********************************************************************/
RTDS_SemaphoreId RTDS_SemCCreate( int maxCount, int initialState )
    {
    RTDS_SemaphoreId SemId;
    
    SemId = ( RTDS_SemaphoreId )RTDS_MALLOC( sizeof( RTDS_SemaphoreInfo ) );
    SemId->semId = CreateSemaphore( NULL, initialState, maxCount, NULL );
    
    if ( SemId->semId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_COUNTING );
        }
    else
        {
        SemId->semType = COUNTING;
        return SemId;
        }
    }

/********************************************************************
 *                   RTDS_SemBCreate
 *-------------------------------------------------------------------
 * Creates a binary semaphore
 *-------------------------------------------------------------------
 * PARAMETERS:
 *      initialState : int  initial state (0: Empty / 1: Full)
 * RETURN:
 *      SemaphoreId : RTDS_SemaphoreId
 *********************************************************************/
RTDS_SemaphoreId RTDS_SemBCreate( int initial_state )
    {
    RTDS_SemaphoreId SemId;
    
    SemId = ( RTDS_SemaphoreId )RTDS_MALLOC( sizeof( RTDS_SemaphoreInfo ) );
    SemId->semId = CreateSemaphore( NULL, initial_state, 1, NULL );
    if ( SemId->semId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_BINARY );
        }
    else
        {
        SemId->semType = BINARY;
        return SemId;
        }
    }

/********************************************************************
 *                       RTDS_SemDelete
 *-------------------------------------------------------------------
 * The purpose of this function is to create binary semaphore
 * (Works for mutex and regular semaphore)
 *-------------------------------------------------------------------
 * PARAMETERS:
 *      semId : RTDS_SemaphoreId
 * RETURN:
 *      Nothing
 *********************************************************************/
void RTDS_SemDelete( RTDS_SemaphoreId semId )
    {
    CloseHandle( semId->semId );
    RTDS_FREE( semId );
    }

/********************************************************************
 *                        RTDS_SemGive
 *-------------------------------------------------------------------
 * The purpose of this function is to release semaphore
 * (Works for mutex and regular semaphore)
 *-------------------------------------------------------------------
 * Assum that the semaphore is a counting (regular semaphore) and try 
 * to increment the counter if operation fails try Mutex command 
 * => if both returns FALSE then return RTDS_ERROR.
 *-------------------------------------------------------------------
 * PARAMETERS:
 *        semId : RTDS_SemaphoreId        semaphore ID
 *********************************************************************/
void RTDS_SemGive( RTDS_SemaphoreId semId )
    {
    /* For mutex or regular semaphore */
    if ( ( semId->semType == COUNTING ) || ( semId->semType == BINARY ) )
        {
        if ( ReleaseSemaphore( semId->semId, 1, NULL ) == 0 )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_SEMAPHORE_GIVE );
            }
        }
    else
        {
        if ( ReleaseMutex( semId->semId ) == 0 )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_SEMAPHORE_GIVE );
            }
        }
    }

/********************************************************************
 *                      RTDS_SemTake
 *-------------------------------------------------------------------
 * Takes a mutex or a regular semaphore
 *-------------------------------------------------------------------
 * PARAMETERS:
 *        semId: RTDS_SemaphoreId    semaphore ID
 *
 * RETURN:
 *        Semaphore status RTDS_OK or RTDS_TIMEOUT
 *********************************************************************/
int RTDS_SemTake( RTDS_SemaphoreId semId, long TIME_TO_WAIT )
    {
    int status;
    
    status = WaitForSingleObject( semId->semId, TIME_TO_WAIT );
    
    if ( status == WAIT_TIMEOUT )
        {
        return RTDS_TIMEOUT;
        }
    else if ( status == WAIT_OBJECT_0 )
        {
        return RTDS_OK;
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEMAPHORE_TAKE );
        }
    }

/********************************************************************
 *                    RTDS_TimerThreadDelete()
 *-------------------------------------------------------------------
 * The purpose of this function is to create Thread based timer
 *-------------------------------------------------------------------
 * Note from MSDN: TerminateThread is used to cause a thread to exit. 
 *  When this occurs, the target thread has no chance to execute any 
 *  user-mode code. DLLs attached to the thread are not notified that 
 *  the thread is terminating. The system frees the thread's initial 
 *  stack.
 *  Windows Server 2003 and Windows XP/2000:  The target thread's 
 *  initial stack is not freed, causing a resource leak !!!
 ********************************************************************/
int RTDS_TimerThreadDelete( HANDLE timerThreadId )
    {
#ifndef RTDS_DISCRETE_TIME /*: Disable timer thread deletion when RTDS_DISCRETE_TIME is not set */
    /* **************************************************************
    Note: Nothing is done because the thread stack is not freed 
    anyway if the thread is deleted from the outside (which is the 
    case here).
    We've better tag the timer to be cancelled so that it exits
    normally (and therefore frees the stack) and is ignored by
    the message loop.
    * ************************************************************** */
    return RTDS_ERROR;
    /*
    if ( TerminateThread( timerThreadId, 0 ) )
        {
        CloseHandle(timerThreadId);
        return RTDS_OK;
        }
    else
        {
        return RTDS_ERROR;
        }
    */
#else
    return RTDS_OK;
#endif
    }

/********************************************************************
 *                    RTDS_TimerThreadFunction()
 *-------------------------------------------------------------------
 * Timer thread function: just sleeps for the delay and calls the
 * watchdog function.
 ********************************************************************/
DWORD RTDS_TimerThreadFunction( LPVOID params )
    {
    RTDS_TimerState * timerState;
    
    timerState = ( RTDS_TimerState * )params;
    
    Sleep( ( long )timerState->timeoutValue - RTDS_GetSystemTime() );
    
    RTDS_WatchDogFunction( timerState );
    
    return RTDS_OK;
    }

/* **************************************************************** *
 *                    RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *     timerList: points to the first element of
 * the chained list of RTDS_TimerState
 * Returns:
 *     The unique timer Id
 * **************************************************************** *
 * Context:
 *     The chained list of RTDS_TimerState is sorted by increasing
 *     unique timer id
 *     timerUniqueId = NULL means the message is not a timer; so it
 *     is avoided.
 * **************************************************************** */
long RTDS_GetTimerUniqueId( RTDS_TimerState * timerList )
    {
    RTDS_TimerState * RTDS_prevTimer;
    RTDS_TimerState * RTDS_timer;
    long newTimerId;
    
    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else if ( timerList->timerUniqueId != 1 )
        {
        newTimerId = 1;
        }
    else
        {
        /* If there is a gap in the timerId chained list, take an Id in the gap */
        RTDS_prevTimer = timerList;
        newTimerId = 0;
        for ( RTDS_timer = timerList->next; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
                {
                if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 )
                    {
                    newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                    }
                RTDS_prevTimer = RTDS_timer;
                }
        /* No gap, let's take the next value */
        if ( newTimerId == 0 )
            {
            newTimerId = RTDS_prevTimer->timerUniqueId + 1;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Check the counter did not go back to 0 */
    if ( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID );
        }
    return newTimerId;
    }

/* **************************************************************** *
 *                      RTDS_WatchDogFunction
 * **************************************************************** *
 * Send a timer message using the parameters given to the watchdog
 * **************************************************************** *
 * Parameters:
 *     parameter: Needed parameters to create and send the timer:
 *         - QueueId of the receiver
 *         - TimerNumber
 *         - TimerUniqueId
 *         - WatchDogId to destroy it
 * Returns:
 *     Nothing
 * **************************************************************** *
 * Context:
 *     Each time a timer is set, a watchdog is created; that's why it
 *     has to destroy itself.
 * **************************************************************** */
void RTDS_WatchDogFunction( RTDS_TimerState * parameter )
    {
    RTDS_MessageHeader * newMessage;
    
    newMessage = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
    
    /* Fill in the message parameters */
#if defined( RTDS_SIMULATOR )
    newMessage->messageUniqueId = 0;
#endif
    newMessage->messageNumber = parameter->timerNumber;
    newMessage->timerUniqueId = parameter->timerUniqueId;
    newMessage->sender = parameter->receiverId;
    newMessage->receiver = parameter->receiverId;
    newMessage->next = NULL;
    newMessage->dataLength = 0;
    newMessage->pData = NULL;
    
    /* Check if process still exist */
    RTDS_CRITICAL_SECTION_START;
    if( RTDS_CheckSdlInstance( parameter->receiverId ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_QUEUEID );
        }
    else
        {
        /* Send the message */
        if ( RTDS_MsgQueueSend( parameter->receiverId->queueId, newMessage ) == RTDS_ERROR )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG );
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_FREE( newMessage );
    }

/* **************************************************************** *
 *                      RTDS_forceTimer
 * **************************************************************** *
 * call RTDS_WatchDogFunction(RTDS_TimerState * parameter)
 * Function called by TCP client to force timer to go off.
 * **************************************************************** *
 * Parameters:
 *         - ProcessId
 *         - TimerName
 * Returns:
 *     Nothing
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) && defined( RTDS_SOCKET_PORT )
void RTDS_ForceTimer( RTDS_SdlInstanceId * sdlInstanceId, int timerNumber )
    {
    RTDS_GlobalProcessInfo  *processInfo;
    RTDS_TimerState         *timerInfo;
    RTDS_TimerState         *timerState = NULL;
    long                    timeOut;
    int                     status;
    
    RTDS_CRITICAL_SECTION_START;
    /* Find timerState (RTDS timer ID) from its timer number and the related processId */
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->mySdlInstanceId == sdlInstanceId )
            {
            for ( timerInfo = processInfo->timerList; timerInfo != NULL; timerInfo = timerInfo->next )
                {
                if ( timerInfo->timerNumber == timerNumber && timerInfo->state == RTDS_TIMER_OK )
                    {
                    timerState = timerInfo;
                    timeOut = timerState->timeoutValue;
                    break;
                    }
                }
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Call timer function to send SDL message */
    if( timerState != NULL )
        {
        /* Let's try to cancel the watchdog */
        status = RTDS_TimerThreadDelete( timerState->watchDogId );
        RTDS_WatchDogFunction( timerState );
    #ifdef RTDS_DISCRETE_TIME
        RTDS_SetSystemTime( timeOut );
    #endif /* RTDS_DISCRETE_TIME */
        }
    }
#endif /* #if defined(RTDS_SIMULATOR) && defined(RTDS_SOCKET_PORT) */

/* **************************************************************** *
 *                      RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *         - QueueId of the receiver
 *         - TimerNumber
 *         - TimerUniqueId
 *         - delay of the timer
 *         - timerStateList: Address of the list of timer
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_StartTimer(
     RTDS_SdlInstanceId      * instanceId,
     long                      timerNumber,
     long                      timerUniqueId,
     int                       delay,
     RTDS_TimerState        ** pTimerStateList,
     RTDS_GlobalProcessInfo  * RTDS_currentContext )
    {
    RTDS_TimerState * timerState;
    RTDS_TimerState * previousTimerState;
    RTDS_TimerState * newTimerState;
    
    /* Update the list of timers */
    /* Important note: it is a sorted list based on the timerUniqueId field */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );
    if ( newTimerState == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MALLOC_TIMER_STATE_IN_START_TIMER );
        }
    
    /* Initialize the new element */
    newTimerState->state            = RTDS_TIMER_OK;
    newTimerState->timerNumber      = timerNumber;
    newTimerState->timeoutValue     = delay + RTDS_GetSystemTime();
    newTimerState->timerUniqueId    = timerUniqueId;
    newTimerState->receiverId       = instanceId;
    newTimerState->next             = NULL;
    
#ifndef RTDS_DISCRETE_TIME
    /* Start a timer thread */
    /* Note: A stack is freed when its thread exits. It is not freed if the thread is terminated by another thread !!! */
    /* So each time a timer is cancelled, the stack for the timer thread is not freed. */
    newTimerState->watchDogId=CreateThread                  /* returns the thread handle */
        (
        NULL,                                               /* default security attributes */
        RTDS_TIMER_TASK_STACK_SIZE,                         /* 1 Mb if not specified */
        ( RTDS_THREAD_START_FUNC )RTDS_TimerThreadFunction, /* thread function */
        ( HANDLE )newTimerState,                            /* argument to thread function */
        0,                                                  /* use default creation flags */
        NULL                                                /* returns the thread identifier */
        );

    if ( newTimerState->watchDogId != NULL )
        {
        ( void )SetThreadPriority( newTimerState->watchDogId, RTDS_TIMER_PRIORITY );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_START );
        }
#else
    /* If simulation mode is on: set Timer threadId to 0 */
    newTimerState->watchDogId = ( HANDLE )0;
#endif
    
    /* Insert the new element */
    if ( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    else    /* The list is not empty */
        {
        previousTimerState = NULL;
        for ( timerState = *pTimerStateList; timerState != NULL; timerState = timerState->next )
            {
            if ( timerState->timerUniqueId > timerUniqueId )
                {
                if ( previousTimerState == NULL )
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }
        if ( timerState == NULL ) /* Inserted at the end of the list */
            {
            previousTimerState->next = newTimerState;
            }
        }
    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState, delay, RTDS_currentContext );
    
    }

/* **************************************************************** *
 *                          RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *         - TimerNumber
 *         - pTimerStateList pointing to the timer chained list
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_StopTimer(
    long timerNumber,
    RTDS_TimerState ** pTimerStateList,
    RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * RTDS_timer;
    RTDS_TimerState * RTDS_prevTimer;
    RTDS_MessageHeader * RTDS_message;
    RTDS_MessageHeader * RTDS_prevMessage;
    
    RTDS_prevTimer = NULL;
    for ( RTDS_timer = *pTimerStateList; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
        {
        if ( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            /* Do not cancel the timer if start transition or if it is the one currently executed */
            if ( ( RTDS_currentContext->currentMessage == NULL ) || 
                ( ( RTDS_currentContext->currentMessage != NULL ) && ( RTDS_currentContext->currentMessage->timerUniqueId != RTDS_timer->timerUniqueId ) ) )
                {
                RTDS_SIMULATOR_TRACE( RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext );
                /* Let's try to cancel the watchdog */
                if ( RTDS_TimerThreadDelete( RTDS_timer->watchDogId ) == RTDS_OK )
                    {
                    /* Remove it from the list */
                    /* Is the first of the list */
                    if ( RTDS_prevTimer == NULL )
                        {
                        *pTimerStateList = RTDS_timer->next;
                        }
                    /* Not the first of the list */
                    else
                        {
                        RTDS_prevTimer->next = RTDS_timer->next;
                        }
                    RTDS_FREE( RTDS_timer );
                    }
                /* Could not cancel the timer. Probably went off allready */
                /* Set it cancelled in the list */
                else
                    {
                    RTDS_timer->state = RTDS_TIMER_CANCELLED;
                    }
                break;
                }
            }
        RTDS_prevTimer = RTDS_timer;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->readSaveQueue; RTDS_message != NULL; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->readSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            break;
            }
        RTDS_prevMessage = RTDS_message;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->writeSaveQueue; RTDS_message != NULL; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            break;
            }
        RTDS_prevMessage = RTDS_message;
        }
    }

/* **************************************************************** *
 *    RTDS_GetProcessQueueId
 * **************************************************************** *
 * Returns the instance Id of process based on its name
 * **************************************************************** *
 * Parameters:
 *         - process name as a number defined in RTDS_gen.h
 * Returns:
 *     the process instance id if found
 * Error:
 *     System error call if not found and NULL returned
 * **************************************************************** */
RTDS_SdlInstanceId * RTDS_GetProcessQueueId( int sdlProcessNumber )
    {
    RTDS_GlobalProcessInfo  * processInfo;
    RTDS_SdlInstanceId      * foundQueue = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == sdlProcessNumber )
            {
            foundQueue = processInfo->mySdlInstanceId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( foundQueue == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_PROCESS_QUEUE_ID );
        }
    return foundQueue;
    }

/* **************************************************************** *
 *    RTDS_CheckSdlInstance
 * **************************************************************** *
 * Returns OK if instance exists or ERROR if not
 * **************************************************************** *
 * Parameters:
 *         - SDL instance id
 * Returns:
 *     RTDS_OK/RTDS_ERROR
 * **************************************************************** */
int RTDS_CheckSdlInstance( RTDS_SdlInstanceId * instanceId )
    {
    RTDS_GlobalProcessInfo * processInfo;
    
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->mySdlInstanceId == instanceId )
            {
            return RTDS_OK;
            }
        }
    return RTDS_ERROR;
    }

/* **************************************************************** *
 *                      RTDS_MsgSend
 * **************************************************************** *
 * Send a message to a process
 * **************************************************************** *
 * Parameters:
 *  - messageNumber         representing a message name
 *  - dataLength            length of data pointed by pData
 *  - pData                 pointer on data sent with the message
 *  - receiver              message receiver queue address
 *  - sender                message sender queue address
 *  - RTDS_currentContext   sending process RTDS context
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_MsgSend(
    long                     messageNumber,
    long                     dataLength,
    unsigned char          * pData,
    RTDS_SdlInstanceId     * receiver,
    RTDS_SdlInstanceId     * sender,
    RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader RTDS_messageToSend;

    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    RTDS_messageToSend.receiver = receiver;
    RTDS_messageToSend.dataLength = dataLength;
    RTDS_messageToSend.pData = pData;
    RTDS_messageToSend.next = NULL;
#if defined( RTDS_SIMULATOR )
    RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
#endif
    
    RTDS_SIMULATOR_TRACE( RTDS_messageSent, &RTDS_messageToSend, receiver, RTDS_currentContext );
    
    RTDS_CRITICAL_SECTION_START;
    /* Check if process still exist */
    if( RTDS_CheckSdlInstance( receiver ) == RTDS_OK )
        {
        if ( RTDS_MsgQueueSend( receiver->queueId, &RTDS_messageToSend )== RTDS_ERROR )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 *                      RTDS_TTCNMsgSend
 * **************************************************************** *
 * Send a message to a TTCN port
 * **************************************************************** *
 * Parameters:
 *  - messageNumber         representing a message name
 *  - dataLength            length of data pointed by pData
 *  - pData                 pointer on data sent with the message
 *  - portId                message receiver queue id
 *  - receiver              message receiver queue address
 *  - sender                message sender queue address
 *  - RTDS_currentContext   sending process RTDS context
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_TTCNMsgSend(
    long                     messageNumber,
    long                     dataLength,
    unsigned char          * pData,
    RTDS_RtosQueueId         portId ,
    RTDS_SdlInstanceId     * receiver,
    RTDS_SdlInstanceId     * sender,
    RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader RTDS_messageToSend;

    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    RTDS_messageToSend.receiver = receiver;
    RTDS_messageToSend.dataLength = dataLength;
    RTDS_messageToSend.pData = pData;
    RTDS_messageToSend.next = NULL;
#if defined( RTDS_SIMULATOR )
    RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
#endif
    
    RTDS_SIMULATOR_TRACE( RTDS_messageSent, &RTDS_messageToSend, receiver, RTDS_currentContext );
    
    RTDS_CRITICAL_SECTION_START;

    if ( RTDS_MsgQueueSend( portId, &RTDS_messageToSend )== RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    RTDS_CRITICAL_SECTION_STOP;
    }
    
/* **************************************************************** *
 *    RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue without waiting for acknoledge
 * if using dynamictrace
 * **************************************************************** *
 * Parameters:
 *         - messageNumber representing a message name
 *         - dataLength length of data pointed by pData
 *         - pData    pointer on data sent with the message
 *         - receiver message receiver queue address
 *         - sender     message sender queue address
 * Returns:
 *     Nothing
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) && defined( RTDS_SOCKET_PORT )
void RTDS_SimulatorMsgQueueSend(
    long                     messageNumber,
    long                     dataLength,
    unsigned char          * pData,
    RTDS_SdlInstanceId     * receiver,
    RTDS_SdlInstanceId     * sender,
    RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader RTDS_messageToSend;
    
    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    RTDS_messageToSend.receiver = receiver;
    RTDS_messageToSend.dataLength = dataLength;
    RTDS_messageToSend.pData = pData;
    RTDS_messageToSend.next = NULL;

    RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
    RTDS_SimulatorTrace( ( enum RTDS_EventType )RTDS_messageSent, ( void * )&RTDS_messageToSend, ( long )receiver, ( RTDS_GlobalProcessInfo * )RTDS_currentContext, RTDS_DTRACE_ACK_NOWAIT );

    if ( RTDS_MsgQueueSend( receiver->queueId, ( RTDS_MessageHeader * ) &RTDS_messageToSend ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }
#endif

/* **************************************************************** *
 *                      RTDS_MsgQueueReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * **************************************************************** *
 * Parameters:
 *         - queue id to read the messsage from
 *         - pointer on the message to receive
 * Returns:
 *     RTDS_OK if everything ok
 * **************************************************************** */
int RTDS_MsgQueueReceive( RTDS_QCB * queueId, RTDS_MessageHeader * message )
    {
    RTDS_MessageHeader * msg;

    #ifdef RTDS_SIMULATOR
      if ( queueId->queue == NULL )
        {
        /* No more messages. Stops if run until rtos message queue empty. */
        RTDS_DummyRunUntilQueueEmptyFunction();
        }
    #endif

    /* Wait for a message to read */
    RTDS_SemTake( queueId->blockingSemId, RTDS_SEMAPHORE_TIME_OUT_FOREVER );
    
    /* Protect Chained list */
    RTDS_SemTake( queueId->chainedListSemId, RTDS_SEMAPHORE_TIME_OUT_FOREVER );
    
    /* Copy message to destination and free memory */
    if ( queueId->queue != NULL )
        {
        msg = queueId->queue;
        memcpy( message, msg, sizeof( RTDS_MessageHeader ) );
        queueId->queue = msg->next;
        RTDS_FREE( msg );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE );
        }
        
    /* Give Chained list semaphore*/
    RTDS_SemGive( queueId->chainedListSemId );
    
    return RTDS_OK;
    }

/* **************************************************************** *
 * RTDS_MsgReceive
 *-------------------------------------------------------------------
 * Receives a message from a process's queue
 *
 * PARAMETERS:
 *   instanceId : RTDS_SdlInstanceId *  Instance id to read the messsage from
 *   message : char *           Pointer on the message to receive
 *   size : int                 Size of message
 *   mode : int                 Mode used to read (blocking or non blocking)
 * **************************************************************** */
void RTDS_MsgReceive( RTDS_SdlInstanceId * instanceId, RTDS_MessageHeader * message )
    {
    if ( RTDS_MsgQueueReceive( instanceId->queueId , message ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE )
        }
    }

/* **************************************************************** *
 * RTDS_TTCNMsgReceive
 *-------------------------------------------------------------------
 * Receives a message from a TTCN queue
 *
 * PARAMETERS:
 *     portId : RTDS_RtosQueueId  Queue id to read the message from
 *     instanceId : RTDS_SdlInstanceId*  Instance id to read the messsage from   
 *     message : char*            Pointer on the message to receive
 *     size : int                 Size of message
 *     mode : int                 Mode used to read (blocking or non blocking)
 * **************************************************************** */
void RTDS_TTCNMsgReceive(RTDS_RtosQueueId portId, RTDS_SdlInstanceId * instanceId, RTDS_MessageHeader *message, int size, int mode)
    {
    if ( RTDS_MsgQueueReceive( portId , message ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE )
        }
    }
    
    
/* *******************************************************************
 * RTDS_TaskCreate
 *-------------------------------------------------------------------
 * Creates a new thread.
 * If any error is encountered RTDS_SYSTEM_ERROR is called and
 * should stop execution.
 *
 * PARAMETERS:
 *     priority : int             Message queue to delete
 *     functionAddress : void*    Adresse of function to execute
 *     newProcessInfo : void*     Parameters for the created thread
 *
 * RETURN:
 *     RTDS_RtosTaskId            ID of the created task 
 ******************************************************************** */
RTDS_RtosTaskId RTDS_TaskCreate(
 int priority ,
 void *( * functionAddress )( void * ) ,
 void * newProcessInfo )
    {
    /* win32 variables */
    DWORD dwThreadId = 1;
    HANDLE hThread;
    
    /* Task creation */
    hThread = CreateThread(
        NULL,                                      /* default security attributes */
        0,                                         /* use default stack size */
        ( RTDS_THREAD_START_FUNC )functionAddress, /* thread function */
        ( HANDLE ) newProcessInfo,                 /* argument to thread function */
        0,                                         /* use default creation flags */
        &dwThreadId );                             /* returns the thread identifier */
    
    /* Check the return value and set priority */
    if ( hThread == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_THREAD );
        }
    else
        {
        if( SetThreadPriority( hThread, priority ) == 0 )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_SET_THREAD_PRIORITY );
            }
        CloseHandle( hThread );         /* Close handle after creating thread */
        }
    return dwThreadId;
    }
/* **************************************************************** *
 *    RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *         - name of the process as a number
 *         - address of the process function
 *         - priority fo the process
 *         - the address of the address of the msg queue to be created
 *         - the address of the msg queue of the parent process
 * Returns:
 *     Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */
void RTDS_ProcessCreate(
    int                      sdlProcessNumber,
    void                   * functionAddress,
    int                      priority,
    RTDS_SdlInstanceId    ** pOffspringId,
    RTDS_SdlInstanceId     * selfId,
    short                    synchronization,
    RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalProcessInfo * processInfo;
    RTDS_GlobalProcessInfo * newProcessInfo;
    
    /* Allocate and fill in a new processInfo structure */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );
    if ( newProcessInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MALLOC );
        }
    
    /* Sets the process name as a number */
    newProcessInfo->sdlProcessNumber = sdlProcessNumber;
    /* SDL initial state */
    newProcessInfo->sdlState = 0;
    
    /* Set SDL task priority */
    newProcessInfo->priority = priority;
    
    /* Last structure of the chained list */
    newProcessInfo->next                    = NULL;
    newProcessInfo->parentSdlInstanceId     = selfId;
    newProcessInfo->offspringSdlInstanceId  = NULL;
    newProcessInfo->currentMessage          = NULL;
    newProcessInfo->timerList               = NULL;
    newProcessInfo->readSaveQueue = NULL;
    newProcessInfo->writeSaveQueue = NULL;
    
    /* New task's queue creation (WINDOWS)*/
    *pOffspringId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
    ( *pOffspringId )->queueId = RTDS_MsgQueueCreate();
    ( *pOffspringId )->instanceNumber = 0;
    newProcessInfo->mySdlInstanceId = *pOffspringId;
    
    /* Take the synchronization semaphore so that the created task does not run straight away */
    if ( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_HOLD;
        }
    
    newProcessInfo->myRtosTaskId = RTDS_TaskCreate( priority ,( void *( * )( void * ) )functionAddress , ( void * )newProcessInfo );
    
    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalProcessInfo == NULL )
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( processInfo = RTDS_globalProcessInfo; processInfo->next != NULL; processInfo = processInfo->next ) ;
        processInfo->next = newProcessInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext );
    
    /* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
    if ( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_UNHOLD;
        }
    }

/* **************************************************************** *
 * RTDS_ProcessForget
 *-------------------------------------------------------------------
 * Forgets a given process, freeing all associated information
 *
 * PARAMETERS:
 *  - RTDS_currentContext : The current context of the task to delete
 *  - deleteQueue: If true, the message queue for the process is destroyed
 * RETURNS:
 *  - The PID for the forgotten process or 0 if process was not found
 * **************************************************************** */
RTDS_RtosTaskId RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext, short deleteQueue )
    {
    RTDS_GlobalProcessInfo  * processInfo;
    RTDS_GlobalProcessInfo  * previousProcessInfo;
    RTDS_TimerState         * timer;
    RTDS_MessageHeader      * message;
    RTDS_MessageHeader      * tmpMessage;
    RTDS_SdlInstanceId      * instance2Delete;
    RTDS_RtosTaskId           pidToKill = 0;
    
    RTDS_CRITICAL_SECTION_START;
    /* Let's free the current message if any */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        #if defined( RTDS_SIMULATOR )
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif
        
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        }
    
    /* Let's clean the timer chained list to free memory and stop watchdogs */
    for ( timer = RTDS_currentContext->timerList ; timer != NULL ; timer = RTDS_currentContext->timerList )
        {
        RTDS_TimerThreadDelete( timer->watchDogId );
        RTDS_currentContext->timerList = timer->next;
        RTDS_FREE( timer );
        }
    
    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    message = RTDS_currentContext->readSaveQueue;
    while( message != NULL )
        {
    #if defined( RTDS_SIMULATOR )
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }
    
    message = RTDS_currentContext->writeSaveQueue;
    while( message != NULL )
        {
    #if defined( RTDS_SIMULATOR )
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }
    
    /* Forget and free the process information in the global instance chained list */
    previousProcessInfo = NULL;
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        /* The queue id is used to find the process id */
        if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )    /* Found ! */
            {
            RTDS_SIMULATOR_TRACE( RTDS_processDied, processInfo, NULL, RTDS_currentContext );
            /* Update the process information chained list */
            if ( previousProcessInfo == NULL )    /* First one in the list */
                {
                RTDS_globalProcessInfo = processInfo->next;
                }
            else
                {
                previousProcessInfo->next = processInfo->next;
                }
            pidToKill = processInfo->myRtosTaskId;
            /* Delete the message queue and free the process info block */
            instance2Delete = processInfo->mySdlInstanceId;
            if ( deleteQueue )
            {
                if ( RTDS_MsgQueueDelete( instance2Delete->queueId ) != RTDS_OK )
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_DELETE );
                    }
            }
            RTDS_FREE( instance2Delete );
            RTDS_FREE( processInfo );
            break;
            }
        previousProcessInfo = processInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Return found PID */
    return pidToKill;
    }
/* **************************************************************** *
 *    RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *         - the current context of the task to delete
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_ProcessKill( RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill;
    
    /* Forget everything about process, delete its message queue and get its PID */
    pidToKill = RTDS_ProcessForget( RTDS_currentContext, 1 );
    
    /* Delete the task */
    if ( pidToKill != 0 )
        {
        /* KILL MACRO WILL CALL 'return' to terminate properly the thread*/
        ExitThread( 0 );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND );
        }
    }

/* **************************************************************** *
 *    RTDS_SemaphoreListInsert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *     - semaphore number
 *     - id of the semaphore
 * Returns:
 *     Nothing
 * **************************************************************** */
RTDS_SemaphoreId RTDS_SemaphoreListInsert(
    int semaphoreNumber,
    RTDS_SemaphoreId semaphoreId,
    RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    long semIsAvailable = -1;
    
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_GlobalSemaphoreInfo * newSemInfo;
    
    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );
    if ( newSemInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT_MALLOC );
        }
    
    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;
    
    /* Semaphore creation */
    newSemInfo->semaphoreId = semaphoreId;
    if ( newSemInfo->semaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT );
        }
    
    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( semInfo = RTDS_globalSemaphoreInfo; semInfo->next != NULL; semInfo = semInfo->next ) ;
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
    semIsAvailable = RTDS_IsSemaphoreAvailable( semaphoreId );
#endif
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated, semaphoreId, semIsAvailable, RTDS_currentContext );
    
    return semaphoreId;
    }

/* **************************************************************** *
 *    RTDS_SemaphoreListRemove
 * **************************************************************** *
 * Kills a semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *         - the address of the semaphore to find the semaphore info
 *         block
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_SemaphoreListRemove(
    RTDS_SemaphoreId semaphoreId,
    RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_GlobalSemaphoreInfo * semInfoPrev;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext );
    
    /* Remove the semaphore information from the chained list */
    semInfoPrev = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreId == semaphoreId ) /* semaphore found */
            {
            if ( semInfoPrev == NULL )
                {
                RTDS_globalSemaphoreInfo = semInfo->next;
                }
            else
                {
                semInfoPrev->next = semInfo->next;
                }
            
            RTDS_SemDelete( semInfo->semaphoreId );
            RTDS_FREE( semInfo );
            break;
            }
        semInfoPrev = semInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 * RTDS_TimerCleanUp
 *-------------------------------------------------------------------
 * Called when the last received message is a timer to figure out if
 * timer was not cancelled before it was received.
 *
 * Parameters:
 *      currentContext : RTDS_GlobalProcessInfo* context of the current task
 * **************************************************************** */
void RTDS_TimerCleanUp( RTDS_GlobalProcessInfo * currentContext )
    {
    RTDS_TimerState * timer;
    RTDS_TimerState * prevTimer;
    
    prevTimer = NULL;
    for ( timer = currentContext->timerList; timer != NULL; timer = timer->next )
        {
        /* If timer found and cancelled */
        if ( timer->timerUniqueId == currentContext->currentMessage->timerUniqueId )
            {
            if ( timer->state == RTDS_TIMER_CANCELLED )
                {
                /* Discard current message */
                RTDS_FREE( currentContext->currentMessage );
                currentContext->currentMessage = NULL;
                }
            
            /* Remove it from list of timers */
            if ( prevTimer == NULL )
                {
                currentContext->timerList = currentContext->timerList->next;
                }
            else
                {
                prevTimer->next = timer->next;
                }
            RTDS_FREE( timer );
            break;
            }
        prevTimer = timer;
        }
    }

/* **************************************************************** *
 *    RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *     None
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo * RTDS_currentContext, int RTDS_sdlStatePrev )
    {
    RTDS_TimerState * RTDS_timer;
    RTDS_TimerState * RTDS_prevTimer;

    /* Free message buffer if valid */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        /* If timer delete corresponding info */
        if ( RTDS_currentContext->currentMessage->timerUniqueId != 0 )
            {
            RTDS_prevTimer = NULL;
            for ( RTDS_timer = RTDS_currentContext->timerList; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
                {
                /* If timer found and cancelled */
                if ( RTDS_timer->timerUniqueId == RTDS_currentContext->currentMessage->timerUniqueId )
                    {
                    /* Remove it from list of timers */
                    if ( RTDS_prevTimer == NULL )
                        {
                        RTDS_currentContext->timerList = RTDS_currentContext->timerList->next;
                        }
                    else
                        {
                        RTDS_prevTimer->next = RTDS_timer->next;
                        }
                    RTDS_FREE( RTDS_timer );
                    break;
                    }
                RTDS_prevTimer = RTDS_timer;
                }
            }
    #ifdef RTDS_SIMULATOR
        /* Release the message unique id back to the pool */
        RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
    #endif
        
        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        RTDS_currentContext->currentMessage = NULL;
        } /* End of if ( RTDS_currentContext->currentMessage != NULL ) */
    
    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if ( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        RTDS_MessageHeader * message;
        /* Let's get to the end of the save queue */
        for ( message=RTDS_currentContext->writeSaveQueue; message->next != NULL; message = message->next );
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    return;
    }

/* **************************************************************** *
 *    RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *     None
 * Returns:
 *     System tick count value
 * **************************************************************** */
unsigned long RTDS_GetSystemTime( void )
    {
#ifndef RTDS_DISCRETE_TIME
    DWORD realTime = 0;
    realTime = GetTickCount();
    RTDS_globalSystemTime = realTime - RTDS_globalStartTime;
    return ( unsigned long )( realTime - RTDS_globalStartTime );
#else
    return RTDS_globalSystemTime;
#endif
    }

/* **************************************************************** *
 *                      RTDS_SetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *     timeValue: long
 * Returns:
 *     None
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) && defined( RTDS_SOCKET_PORT )
void RTDS_SetSystemTime( long timeValue )
    {
#ifndef RTDS_DISCRETE_TIME
    unsigned long realTime = 0;
    realTime = ( unsigned long )GetTickCount();
    RTDS_globalStartTime = realTime - ( unsigned long )timeValue;
#endif
    RTDS_globalSystemTime = timeValue;
    }
#endif

/* **************************************************************** *
 *                  RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *     - the id of the semaphore
 *     - timeout value     -> NOT USED IN THIS PROFILE
 * Returns:
 *     - status : RTDS_OK or RTDS_TIMEOUT
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_SemaphoreIdTake(
    RTDS_SemaphoreId semaphoreId,
    RTDS_SemaphoreTimeout TIME_OUT,
    RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_SemaphoreStatus tmpStatus;
    long semIsAvailable = -1;
    
    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt, semaphoreId, TIME_OUT, RTDS_currentContext );
    
    tmpStatus = RTDS_SemTake( semaphoreId, TIME_OUT );
    if ( tmpStatus == RTDS_OK )
        {
    #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
        semIsAvailable=RTDS_IsSemaphoreAvailable(semaphoreId);
    #endif
        RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded, semaphoreId, semIsAvailable, RTDS_currentContext );
        }
    else if ( tmpStatus == RTDS_TIMEOUT )
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeTimedOut, semaphoreId, 0, RTDS_currentContext );
        }
    return tmpStatus;
    }

/* **************************************************************** *
 *    RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *     - semaphore number representing its name
 * Returns:
 *     - the id of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_SemaphoreId foundSemaphoreId;
    
    foundSemaphoreId = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( foundSemaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID );
        }
    
    return foundSemaphoreId;    /* Might cause an RTOS exception if NULL*/
    }

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */
#if defined( RTDS_SIMULATOR )
void * RTDS_globalMessageUniqueIdPool = NULL;
#endif

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError, NULL, 0 };
char * RTDS_globalPrintableParameters = NULL;
#endif

#if defined( RTDS_SIMULATOR )
/* **************************************************************** *
 *    RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *     None
 * Returns:
 *     - the message unique id (minimum is 1)
 * **************************************************************** */
unsigned long RTDS_GetMessageUniqueId( void )
    {
    unsigned char * index;
    long uniqueByteId;
    long uniqueBitId;
    
    index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
    
    RTDS_CRITICAL_SECTION_START;
    for ( uniqueByteId = 0; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE; uniqueByteId++ )
        {
        if( *index != 0xFF )
            {
            for( uniqueBitId = 0; uniqueBitId < 8; uniqueBitId++ )
                {
                if ( ( ( 1 << uniqueBitId ) & *index ) == 0 )
                    {
                    *index = *index | ( 1 << uniqueBitId );
                    RTDS_CRITICAL_SECTION_STOP;
                    return ( 8 * uniqueByteId + uniqueBitId + 1 );
                    }
                }
            }
        index++;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* All bits are set... No more message unique id */
    RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID );
    return 0;
    }

/* **************************************************************** *
 *    RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *     message unique id
 * Returns:
 *     nothing
 * **************************************************************** */
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
    {
    unsigned char * index;
    
    if ( messageUniqueId == 0 ) /* probably a timer */
        {
        return;
        }
    messageUniqueId -= 1;
    index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
    index += ( unsigned char )( messageUniqueId / 8 );
    RTDS_CRITICAL_SECTION_START;
    ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
    RTDS_CRITICAL_SECTION_STOP;
    }

#endif /* #if defined(RTDS_SIMULATOR) */

/* **************************************************************** *
 *    RTDS_SimulatorTrace
 * **************************************************************** *
 * As its name states... Handles trace with the host depending on
 * the type of link available and options set in the profile.
 * **************************************************************** *
 * Parameters:
 *    enum RTDS_EventType  event:                         trace event
 *    void *eventParameter1:                                    parameter 1 of the trace
 *    long eventParameter2:                                     parameter 2 of the trace
 *    RTDS_GlobalProcessInfo *currentContext: calling context
 * Returns:
 *     nothing
 * **************************************************************** */
/* Function defined is trace needed */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )

void RTDS_SimulatorTrace( enum RTDS_EventType event, void * eventParameter1, long eventParameter2, RTDS_GlobalProcessInfo * currentContext, int waitAck )
    {
#ifdef RTDS_FORMAT_TRACE
    int formatStatus = RTDS_ERROR;
    char * formattedCmd = NULL;
#endif /* RTDS_FORMAT_TRACE */
    
    unsigned long sysTime = 0;
    sysTime = RTDS_GetSystemTime();
    
    if ( waitAck == RTDS_DTRACE_ACK_WAIT )
        {
        RTDS_CRITICAL_TRACE_SECTION_START;
        }
    
    /* Format trace */
    RTDS_globalTraceEntry.event = event;
    RTDS_globalTraceEntry.eventParameter1 = ( void * )eventParameter1;
    RTDS_globalTraceEntry.eventParameter2 = ( long )eventParameter2;
    RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )currentContext;

#ifdef RTDS_FORMAT_TRACE
    formatStatus = RTDS_FormatTrace( sysTime, RTDS_globalTraceEntry, waitAck, &formattedCmd );
    /* Send trace to RTDS or MSC Tracer */
#if defined( RTDS_SOCKET_PORT )
    if ( formattedCmd != NULL )
        {
        RTDS_SOCKET_ACCESS_TAKE;
        RTDS_SendSocket( globalClientSocketId, formattedCmd, strlen( formattedCmd ) );
        RTDS_SOCKET_ACCESS_GIVE;
        if ( waitAck == RTDS_DTRACE_ACK_WAIT )
            { 
            RTDS_DTRACE_ACKNOWLEDGE_WAIT; 
            }
        }
#endif

#ifdef RTDS_SIMULATOR
    /* Called for backtrace*/
    RTDS_DummyTraceFunction();
#endif

    RTDS_FREE( formattedCmd );

#else /* not RTDS_FORMAT_TRACE, just RTDS_SIMULATOR*/
    if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
        {
        RTDS_messageDataToString( &RTDS_globalPrintableParameters, ( ( RTDS_MessageHeader * )eventParameter1 )->messageNumber, ( ( RTDS_MessageHeader * )eventParameter1 )->dataLength, ( void * )( ( ( RTDS_MessageHeader * )eventParameter1 )->pData ), RTDS_PARAM_CODEC_MAX_DEPTH );
        }
    RTDS_DummyTraceFunction();
    if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
        {
        RTDS_FREE( RTDS_globalPrintableParameters );
        RTDS_globalPrintableParameters = NULL;
        }
#endif
    
    if ( waitAck == RTDS_DTRACE_ACK_WAIT )
        {
        RTDS_CRITICAL_TRACE_SECTION_STOP;
        }
    }

#endif /* defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER) */

/* **************************************************************** *
 *    RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *     None
 * Returns:
 *     nothing
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
void RTDS_DummyTraceFunction( void )
    {
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
    RTDS_TraceAdd();
#endif
    }
#endif /* RTDS_SIMULATOR */


/* **************************************************************** *
 *    RTDS_DummyRunUntilQueueEmptyFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function when the user asks tu run until the queue is empty. 
 * Whenever the system stops the breakpoint is removed.
 * **************************************************************** *
 * Parameters:
 *     None
 * Returns:
 *     nothing
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
void RTDS_DummyRunUntilQueueEmptyFunction( void )
    {
    }
#endif /* RTDS_SIMULATOR */


#if defined( RTDS_FORMAT_TRACE ) || defined( RTDS_SIMULATOR )
/* **************************************************************** *
 *    RTDS_GetProcessInfoFromSdlInstanceId
 * **************************************************************** *
 * Returns the current RTDS_GlobalProcessInfo
 * **************************************************************** *
 * Parameters:
 *     RTDS_SdlInstanceId *
 * Returns:
 *     RTDS_GlobalProcessInfo *
 * **************************************************************** */
RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromSdlInstanceId( RTDS_SdlInstanceId * sdlInstanceId )
    {
    RTDS_GlobalProcessInfo * processInfo;

    if ( RTDS_globalProcessInfo == NULL )
        {
        return RTDS_globalProcessInfo;
        }
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->mySdlInstanceId == sdlInstanceId )
            {
            return processInfo;
            }
        }

    return NULL; /* If not found */
    }

/* **************************************************************** *
 *    RTDS_GetProcessInfoFromRtosTaskId
 * **************************************************************** *
 * Returns the current RTDS_GlobalProcessInfo
 * **************************************************************** *
 * Parameters:
 *     RTDS_RtosTaskId
 * Returns:
 *     RTDS_GlobalProcessInfo *
 * Use:
 *    presently used by RTDS_SYSTEM_ERROR macro
 * **************************************************************** */
RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromRtosTaskId( RTDS_RtosTaskId rtosTaskId )
    {
    RTDS_GlobalProcessInfo * processInfo;

    if ( RTDS_globalProcessInfo == NULL )
        {
        return RTDS_globalProcessInfo;
        }
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->myRtosTaskId == rtosTaskId )
            {
            return processInfo;
            }
        }

    return NULL; /* If not found */
    }

#endif

/* **************************************************************** *
 *    RTDS_GetProcessInfoFromProcessNum
 * **************************************************************** *
 * Returns the RTDS_GlobalProcessInfo of the corresponding process number
 * **************************************************************** *
 * Parameters:
 *     int sdlProcessNumber
 * Returns:
 *     RTDS_GlobalProcessInfo *
 * Use:
 *    presently used by RTDS_SYSTEM_ERROR macro
 * **************************************************************** */
#if defined( RTDS_FORMAT_TRACE ) || defined( RTDS_SIMULATOR )
RTDS_GlobalProcessInfo * RTDS_GetProcessInfoFromProcessNum( int sdlProcessNumber )
    {
    RTDS_GlobalProcessInfo * processInfo;
    
    if ( RTDS_globalProcessInfo == NULL )
        {
        return RTDS_globalProcessInfo;
        }

    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == sdlProcessNumber )
            {
            return processInfo;
            }
        }
    return NULL; /* If not found */
    }
#endif

/* **************************************************************** *
 *    RTDS_GetCurrentRtosTaskId
 * **************************************************************** *
 * Returns the current RTDS_RtosTaskId
 * **************************************************************** *
 * Parameters:
 *     none
 * Returns:
 *     RTDS_RtosTaskId
 * Use:
 *    Presently used by RTDS_SYSTEM_ERROR macro
 * **************************************************************** */
#if defined( RTDS_FORMAT_TRACE ) || defined( RTDS_SIMULATOR )
RTDS_RtosTaskId RTDS_GetCurrentRtosTaskId( void )
    {
    return GetCurrentThreadId();
    }
#endif

/* **************************************************************** *
 *    RTDS_IsSemaphoreAvailable
 * **************************************************************** *
 * Returns the semaphore state
 * **************************************************************** *
 * Parameters:
 *     RTDS_SemaphoreId semaphoreId
 * Returns:
 *     int: -1 if function not available
 *                 0 if semaphore not available
 *                 1 if semaphore available
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
long RTDS_IsSemaphoreAvailable( RTDS_SemaphoreId semaphoreId )
    {
    int stillAvailable = -1;
    if ( RTDS_SemTake( semaphoreId, RTDS_SEMAPHORE_TIME_OUT_NO_WAIT ) == RTDS_TIMEOUT )
        {
        stillAvailable = 0;
        }
    else
        {
        RTDS_SemGive( semaphoreId );
        stillAvailable = 1;
        }
    return stillAvailable;
    }
#endif


