/* PragmaDev RTDS ThreadX integration */
#ifndef _RTDS_BASIC_TYPES_H_
#define _RTDS_BASIC_TYPES_H_
#ifdef __cplusplus
extern "C" {
#endif

#include "stdio.h"
#include "stdlib.h"

/* ThreadX include */
#include	 "tx_api.h"

/* Include for RTDS types */
#include "RTDS_Common.h"

/*
 * DEFINE FOR MEMORY MANIPULATION:
 * ------------------------------
 */
#define RTDS_THREADX_BYTE_POOL_SIZE 		0x10000 				 /* Size of memory pool for all dynamic memory allocation */


/*
 * DEFINE FOR QUEUE MANIPULATION:
 * ------------------------------
 */
#define RTDS_QUEUE_MAX_MSG							20							 /* Maximum number of messages in a queue */
#define RTDS_QUEUE_MAX_MSG_LENGTH 			TX_1_ULONG			 /* Maximum length of messages in a queue: one pointer... */
#define RTDS_QUEUE_WAIT_FOREVER 				TX_WAIT_FOREVER

/*
 * DEFINE FOR SEMAPHORE MANIPULATION:
 * ----------------------------------
 */
#define RTDS_SEMAPHORE_MAX_NAME_LENGTH	20							 /* Maximum length of a semaphore name */

#define RTDS_SEMAPHORE_TIME_OUT_FOREVER TX_WAIT_FOREVER  /* Time out option when taking the semaphore : wait forever */
#define RTDS_SEMAPHORE_TIME_OUT_NO_WAIT TX_NO_WAIT			 /* Time out option when taking the semaphore : no wait */

#define RTDS_SEMAPHORE_OPTION_PRIO			0 							 /* Semaphore creation option: queuing based on priority */
#define RTDS_SEMAPHORE_OPTION_FIFO			1 							 /* Semaphore creation option: queuing based on FIFO */

#define RTDS_BINARY_SEM_INITIAL_EMPTY 	0 							 /* Binary semaphore creation initial state: empty */
#define RTDS_BINARY_SEM_INITIAL_FULL		1 							 /* Binary semaphore creation initial state: full */

#define RTDS_MUTEX_SEM_DELETE_SAFE			0 							 /* Protects a task that owns the semaphore from unexpected deletion */
#define RTDS_MUTEX_SEM_INVERSION_SAFE 	1 							 /* Protect the system from priority inversion */

/*
 * DEFINE FOR PROCESS CREATION:
 * ----------------------------
 */
#define RTDS_TASK_OPTIONS 						0 								/* Not used with ThreadX */
#define RTDS_TASK_STACK_SIZE					0x1000 						/* Stack size when creating a new task */
#define RTDS_DEFAULT_PROCESS_PRIORITY 15								/* If priority not defined in SDL-RT, it will be given this value */


/* ************************************************************************************ *
 *									CRITICAL SECTION MACROS
 * ************************************************************************************ */
#define RTDS_CRITICAL_SECTION_DECL				RTDS_MutexId RTDS_globalSystemSemId=NULL;
#define RTDS_CRITICAL_SECTION_PROTO 			extern RTDS_MutexId RTDS_globalSystemSemId;
#define RTDS_CRITICAL_SECTION_INIT				tx_mutex_create( RTDS_globalSystemSemId, "RTDS_CriticalSection", TX_NO_INHERIT);
#define RTDS_CRITICAL_SECTION_START 			tx_mutex_get( RTDS_globalSystemSemId, TX_WAIT_FOREVER);
#define RTDS_CRITICAL_SECTION_STOP				tx_mutex_put( RTDS_globalSystemSemId );
#define RTDS_CRITICAL_SECTION_POSTAMBLE 	tx_mutex_delete( RTDS_globalSystemSemId );


/* ************************************************************************************ *
 *									COVERAGE INFORMATION
 * ************************************************************************************ *
 * The coverage information is stored in an array of unsigned char. The SDL-RT symbol
 * number is used as an index; the value stored is the number of times the symbol has
 * been executed. It can not exceed 0xFF for each symbol.
 * ************************************************************************************ */
#ifdef RTDS_COVERAGE_NB_SYMBOLS
#define RTDS_COVERAGE_DECL		unsigned char  RTDS_globalCoverage[RTDS_COVERAGE_NB_SYMBOLS];
#define RTDS_COVERAGE_PROTO 	extern RTDS_COVERAGE_DECL
#define RTDS_COVERAGE_INIT \
	{ \
	int RTDS_i; \
	for (RTDS_i=0;RTDS_i<RTDS_COVERAGE_NB_SYMBOLS;RTDS_i++) \
		RTDS_globalCoverage[RTDS_i]=0; \
	}
#define RTDS_COVERAGE_LOG(SYMBOL_NUMBER) \
	if (RTDS_globalCoverage[SYMBOL_NUMBER]!=0xFF) RTDS_globalCoverage[SYMBOL_NUMBER]++;
#else
#define RTDS_COVERAGE_DECL
#define RTDS_COVERAGE_PROTO
#define RTDS_COVERAGE_INIT
#endif


/*
 * DEFINE FOR SIMULATION:
 * ------------------------------
 * Define on how many bytes the messageUniqueId pool will be made of
 * 1 means 1 byte means 1*8 bits means 8 possible simultaneous values
 * Maximum is 8192 since the id a long type
 */
#ifdef RTDS_SIMULATOR
	/* Define on how many bytes the messageUniqueId pool will be made of
	 * 1 means 1 byte means 1*8 bits means 8 possible simultaneous values
	 * Maximum is 8192 since the id a long type */
	#define RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE	64
	/* Defines when tracing through socket to synchronize target execution and host debugger */
	#define RTDS_DTRACE_ACK_NOWAIT						0 	/* No acknowledgment from the host */
	#define RTDS_DTRACE_ACK_WAIT							1 	/* Ack from the host */
	/* Number of levels when formatting message parameters to be printable
	 * in the SDL-RT debugger */
	#define RTDS_PARAM_CODEC_MAX_DEPTH				4
	/* Min size of memory chunk used when formatting message parameters
	 * to be printable in the SDL-RT debugger */
	#define RTDS_PARAM_CODEC_CHUNK_SIZE 			128
#endif

/*
 * TYPE RTDS_QUEUE.ID:
 * ----------------------
 * Type for a message queue identifier
 */
typedef TX_QUEUE *RTDS_RtosQueueId;


/*
 * TYPE RTDS_SEMAPHORE.ID:
 * ----------------------
 * Type for a semaphore identifier
 */
typedef TX_SEMAPHORE *RTDS_SemaphoreId;

/*
 * TYPE RTDS_MUTEX.ID:
 * ----------------------
 * Type for a mutex semaphore identifier
 */
typedef TX_MUTEX *RTDS_MutexId;

/*
 * TYPE RTDS_SEMAPHORE_STATUS:
 * ----------------------
 * Type for a semaphore status
 */
typedef UINT RTDS_SemaphoreStatus;

#define RTDS_OK 		TX_SUCCESS
#define RTDS_ERROR	0xFF

/*
 * TYPE RTDS_SEMAPHORE_TIMEOUT:
 * ----------------------
 * Type for a semaphore timeout value
 */
typedef ULONG RTDS_SemaphoreTimeout;

/*
 * TYPE RTDS_PROCESS_ID:
 * ----------------------
 * Type for a pid
 */
typedef TX_THREAD *RTDS_RtosTaskId;

/*
 * TYPE RTDS_PROCESS_PRIORITY:
 * ----------------------
 * Type for process priority
 */
typedef UINT RTDS_ProcessPriority;


/*
 * STRUCT RTDS_TIMER.STATE:
 * ------------------------
 * Element in chained list for all active timers
 */
#define RTDS_TIMER_OK 				1
#define RTDS_TIMER_CANCELLED	0

typedef struct RTDS_TimerState
	{
	long										  timerNumber;			/* Message number for the timer name */
	long										  timerUniqueId;		/* Timer's unique identifier among all timers */
	unsigned long 					  timeoutValue; 		/* System tick counter value when it will go off */
	struct RTDS_SdlInstanceId *receiverId;    	/* The message queue of the receiver */
	int 										  state;						/* State for the timer (RTDS_TIMER_OK or RTDS_TIMER_CANCELLED) */
	TX_TIMER								  *watchDogId;			/* Id of the timer */
	struct RTDS_MessageHeader	*messageAddress;	/* The timer message allocated when starting the timer */

	struct RTDS_TimerState	*next;	/* Next timer in list */
	} RTDS_TimerState;



/*
 * ENUM RTDS_SEMAPHORE.TYPE:
 * -------------------------
 * Enum of semaphore types that can be manipulated in SDL-RT
 */

enum RTDS_SemaphoreType
	{
	RTDS_binary 	 = 0,
	RTDS_counting  = 1,
	RTDS_mutex		 = 2
	} ;



/*
 * STRUCT RTDS_GLOBAL.SEMAPHORE.INFO:
 * -------------------------------
 * Type for the list of semaphores in the system with their information
 */
typedef struct RTDS_GlobalSemaphoreInfo
	{
	RTDS_SemaphoreId								semaphoreId;			/* The semaphore address */
	enum RTDS_SemaphoreType 				semaphoreType;		/* Semaphore type */
	int 														semaphoreNumber;	/* Semaphore number representing its name */
	struct RTDS_GlobalSemaphoreInfo *next;						/* Next semaphoreInfo */
	} RTDS_GlobalSemaphoreInfo;

/*
 * Additional fields in common types
 * ---------------------------------
 */
/* Message header: none */
#define RTDS_MESSAGE_HEADER_ADDITIONNAL_FIELDS
/* Process information: add priority */
#define RTDS_GLOBAL_PROCESS_INFO_ADDITIONNAL_FIELDS \
  RTDS_ProcessPriority  priority; \
  void                  *pStack; \
  void                  *pQueue;


#ifdef __cplusplus
}
#endif

#endif /* _RTDS_BASIC_TYPES_H_ */
