/* PragmaDev RTDS OSE 52 integration */

#include "RTDS_MACRO.h"
#include "efs.h"

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE ) || defined( RTDS_SOCKET_IP_ADDRESS )
#include "RTDS_Trace.h"
#endif

/* **************************************************************** *
 * RTDS_MemAlloc
 * **************************************************************** *
 * Allocate a shared amount of memory
 * **************************************************************** *
 * Parameters:
 *      size: amount of memory required
 * Returns:
 *      Pointer to the allocated memory
 * **************************************************************** *
 * Context:
 *      Because a process can free a bloc of memory allocated for 
 *      another process. We have to share allocated memory
 * **************************************************************** */
void * RTDS_MemAlloc( int size )
    {
    return heap_alloc_shared( size, NULL, 0 );
    }
 
/* **************************************************************** *
 * RTDS_Free
 * **************************************************************** *
 * Free a shared memory
 * **************************************************************** *
 * Parameters:
 *      Pointer to the bloc of memory to release
 * Returns:
 *      Nothing
 * **************************************************************** *
 * Context:
 *      Because a process can free a bloc of memory allocated for 
 *      another process, we have to call this primitive
 * **************************************************************** */
void RTDS_Free( void * ptr )
    {
    heap_free_shared( ptr );
    }

/* **************************************************************** *
 * RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *      timerList: points to the first element of
 * the chained list of RTDS_TimerState
 * Returns:
 *      The unique timer Id
 * **************************************************************** *
 * Context:
 *      The chained list of RTDS_TimerState is sorted by increasing
 *      unique timer id
 *      timerUniqueId = NULL means the message is not a timer; so it
 *      is avoided.
 * **************************************************************** */
long RTDS_GetTimerUniqueId( RTDS_TimerState * timerList )
    {
    long newTimerId;
    RTDS_TimerState * RTDS_prevTimer;
    RTDS_TimerState * RTDS_timer;
    
    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if ( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else if ( timerList->timerUniqueId != 1 )
        {
        newTimerId = 1;
        }
    else
        {
        /* If there is a gap in the timerId chained list, take an Id in the gap */
        RTDS_prevTimer = timerList;
        newTimerId = 0;
        for ( RTDS_timer = timerList->next; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
            {
            if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 ) 
                {
                newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                }
            RTDS_prevTimer = RTDS_timer;
            }
        /* No gap, let's take the next value */
        if ( newTimerId == 0 ) 
            {
            newTimerId = RTDS_prevTimer->timerUniqueId + 1;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Check the counter did not go back to 0 */
    if ( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID );
        }
    return newTimerId;
    }

/* **************************************************************** *
 * RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *      - InstanceId of the receiver
 *      - TimerNumber
 *      - TimerUniqueId
 *      - delay of the timer
 *      - timerStateList: Address of the list of timer
 *      - currentContext (RTDS_GlobalProcessInfo *): active task context
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_StartTimer
( RTDS_SdlInstanceId * instanceId,
 long timerNumber,
 long timerUniqueId,
 int delay,
 RTDS_TimerState ** pTimerStateList,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader * sig;
    RTDS_TimerState * timerState;
    RTDS_TimerState * newTimerState;
    RTDS_TimerState * previousTimerState;
    
    /* Create a timer message */
    sig = ( RTDS_MessageHeader * )alloc( sizeof( RTDS_MessageHeader ), timerNumber );
    sig->messageNumber = timerNumber;
    
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
    sig->messageUniqueId = 0;
#endif
    
    sig->timerUniqueId = timerUniqueId;
    sig->sender = instanceId;
    sig->dataLength = 0;
    sig->pData = NULL;
    sig->next = NULL;
    
    /* Update the list of timers */
    /* Important note: it is a sorted list based on the timerUniqueId field */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );
    if ( newTimerState == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MALLOC_TIMER_STATE_IN_START_TIMER );
        }
    
    /* Initialize the new element */
    newTimerState->state = RTDS_TIMER_OK;
    newTimerState->timerNumber = timerNumber;
    newTimerState->timeoutValue = delay + get_ticks();
    newTimerState->timerUniqueId = timerUniqueId;
    newTimerState->next = NULL;
    newTimerState->receiverQueueId = instanceId;
    
    /* Start the timer */
    requestTmo( &( newTimerState->watchDogId ), delay,( union SIGNAL ** )( &sig ) );
    
    /* Insert the new element */
    if ( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    /* The list is not empty */
    else
        {
        previousTimerState = NULL;
        for ( timerState = *pTimerStateList; timerState != NULL; timerState = timerState->next )
            {
            if ( timerState->timerUniqueId > timerUniqueId )
                {
                if ( previousTimerState == NULL )
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }
        if ( timerState == NULL ) /* Inserted at the end of the list */
            {
            previousTimerState->next = newTimerState;
            }
       }
    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState, delay, RTDS_currentContext );
    }

/* **************************************************************** *
 * RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *      - TimerNumber
 *      - pTimerStateList pointing to the timer chained list
 *      - RTDS_currentContext currentContext of the task
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_StopTimer
( long timerNumber,
 RTDS_TimerState ** pTimerStateList,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    union SIGNAL * timerSignal = NULL;
    RTDS_TimerState * RTDS_timer = NULL;
    RTDS_TimerState * RTDS_prevTimer = NULL;
    RTDS_MessageHeader * RTDS_message = NULL;
    RTDS_MessageHeader * RTDS_prevMessage = NULL;
    
    for ( RTDS_timer = *pTimerStateList; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
        {
        if ( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            RTDS_SIMULATOR_TRACE( RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext );
            
            /* Let's try to cancel the watchdog */
            timerSignal = cancelTmo( &( RTDS_timer->watchDogId ) );
            if ( timerSignal != 0 )
                {
                /* Anyway Remove it from the list */
                /* Is the first of the list */
                if ( RTDS_prevTimer == NULL )
                    {
                    *pTimerStateList = RTDS_timer->next;
                    }
                /* Not the first of the list */
                else
                    {
                    RTDS_prevTimer->next = RTDS_timer->next;
                    }
                RTDS_FREE(RTDS_timer);
                
                /* Free the timer signal */
                free_buf( &timerSignal );
                }
            /* Could not cancel the timer. Probably went off allready */
            /* Set it cancelled in the list */
            else
                {
                RTDS_timer->state = RTDS_TIMER_CANCELLED;
                }
            return;
            }
        RTDS_prevTimer = RTDS_timer;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->readSaveQueue; RTDS_message != NULL; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->readSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            /* Free memory */
            free_buf( ( union SIGNAL ** ) &( RTDS_message ) );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->writeSaveQueue; RTDS_message != NULL; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            /* Free memory */
            free_buf( ( union SIGNAL ** ) &( RTDS_message ) );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    }

/* **************************************************************** *
 * RTDS_GetProcessQueueId
 * **************************************************************** *
 * Returns the queueId of process based on its name
 * **************************************************************** *
 * Parameters:
 *      process name as a number defined in RTDS_gen.h
 * Returns:
 *      the process queue id if found
 * Error:
 *      System error call if not found and NULL returned
 * **************************************************************** */
RTDS_SdlInstanceId * RTDS_GetProcessQueueId( int processNumber )
    {
    RTDS_SdlInstanceId * foundQueue = NULL;
    RTDS_GlobalProcessInfo * processInfo = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == processNumber )
            {
            foundQueue = processInfo->mySdlInstanceId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( foundQueue == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_PROCESS_QUEUE_ID );
        }
    return foundQueue;
    }

/* **************************************************************** *
 * RTDS_MsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *      - messageNumber representing a message name
 *      - dataLength length of data pointed by pData
 *      - pData pointer on data sent with the message
 *      - receiver message receiver queue address
 *      - sender message sender queue address
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_MsgQueueSend
( long messageNumber,
 long dataLength,
 unsigned char * pData,
 RTDS_SdlInstanceId * receiver,
 RTDS_SdlInstanceId * sender,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader * RTDS_messageToSend = NULL;
    
    RTDS_messageToSend = ( RTDS_MessageHeader * )alloc( sizeof( RTDS_MessageHeader ), messageNumber );
    RTDS_messageToSend->messageNumber = messageNumber;
    RTDS_messageToSend->timerUniqueId = 0;
    RTDS_messageToSend->sender = sender;
    RTDS_messageToSend->receiver = receiver;
    RTDS_messageToSend->dataLength = dataLength;
    RTDS_messageToSend->pData = pData;
    RTDS_messageToSend->next = NULL;
    
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
    RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
    RTDS_messageDataToString( &RTDS_globalPrintableParameters, messageNumber, dataLength, ( void * )pData, RTDS_PARAM_CODEC_MAX_DEPTH );
    RTDS_SIMULATOR_TRACE( RTDS_messageSent, RTDS_messageToSend, receiver, RTDS_currentContext );
    RTDS_FREE( RTDS_globalPrintableParameters );
    RTDS_globalPrintableParameters = NULL;
#endif
    
    send( ( union SIGNAL ** )( &RTDS_messageToSend ), receiver->queueId );
    }

/* **************************************************************** *
 * RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *      - name of the process as a char string
 *      - name of the process as a number
 *      - address of the process function
 *      - priority fo the process
 *      - the address of the address of the msg queue to be created
 *      - the address of the msg queue of the parent process
 *      - RTDS_currentContext currentContext of the task
 * Returns:
 *      - Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */
void RTDS_ProcessCreate
( char * processName,
 int processNumber,
 void * functionAddress,
 int priority,
 RTDS_SdlInstanceId ** pOffspringId,
 RTDS_SdlInstanceId * selfId,
 short synchronization,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalProcessInfo * processInfo = NULL;
    RTDS_GlobalProcessInfo * newProcessInfo = NULL;
    
    /* Allocate and fill in a new processInfo structure */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );
    if ( newProcessInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MALLOC );
        }
    
    /* Sets the process name as a number */
    newProcessInfo->sdlProcessNumber = processNumber;
    /* SDL initial state */
    newProcessInfo->sdlState = 0;
    /* Last structure of the chained list */
    newProcessInfo->mySdlInstanceId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
    newProcessInfo->next = NULL;
    newProcessInfo->parentSdlInstanceId = selfId;
    newProcessInfo->offspringSdlInstanceId = NULL;
    newProcessInfo->currentMessage = NULL;
    newProcessInfo->timerList = NULL;
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
    newProcessInfo->priority = priority;
#endif
    
    /* Task creation */
    /* Please note error handling is implicit */
    newProcessInfo->myRtosTaskId = create_process
    ( OS_PRI_PROC,
    processName,
    ( OSENTRYPOINT * )functionAddress,
    RTDS_TASK_STACK_SIZE,
    ( OSPRIORITY ) priority,
    ( OSTIME )0,
    ( RTDS_RtosTaskId )0,
    ( struct OS_redir_entry * )NULL,
    ( OSVECTOR )0,
    ( OSUSER )0 );
    
    /* share all file descriptor */
    efs_clone( newProcessInfo->myRtosTaskId );
    
    /* The queue id can be considered to be the pid because the queue is implicit in OSE */
    newProcessInfo->mySdlInstanceId->queueId = newProcessInfo->myRtosTaskId;
    *pOffspringId = newProcessInfo->mySdlInstanceId;

    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalProcessInfo == NULL )
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( processInfo = RTDS_globalProcessInfo; processInfo->next != NULL; processInfo = processInfo->next );
        processInfo->next = newProcessInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext );
    /* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
    if ( synchronization == RTDS_NO_HOLD )
        {
        start( newProcessInfo->myRtosTaskId );
        }
    }

/* **************************************************************** *
 * RTDS_ProcessForget
 *-------------------------------------------------------------------
 * Forgets a given process, freeing all associated information
 *
 * PARAMETERS:
 *  - RTDS_currentContext : The current context of the task to delete
 *  - deleteQueue: If true, the message queue for the process is destroyed
 * RETURNS:
 *  - The PID for the forgotten process or 0 if process was not found
 * **************************************************************** */
RTDS_RtosTaskId RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill = 0;
    RTDS_TimerState * RTDS_timer = NULL;
    RTDS_MessageHeader * message = NULL;
    RTDS_MessageHeader * tmpMessage = NULL;
    RTDS_GlobalProcessInfo * processInfo = NULL;
    RTDS_GlobalProcessInfo * previousProcessInfo = NULL;

    /* Let's free the current message if any */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free memory */
        free_buf( ( union SIGNAL ** ) &( RTDS_currentContext->currentMessage ) );
        }
    
    /* Let's clean the timer chained list to free memory and stop watchdogs */
    for ( RTDS_timer = RTDS_currentContext->timerList; RTDS_timer != NULL; RTDS_timer = RTDS_currentContext->timerList )
        {
        RTDS_currentContext->timerList = RTDS_timer->next;
        RTDS_FREE( RTDS_timer );
        }
    
    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    for ( message = RTDS_currentContext->readSaveQueue; message!=NULL; )
        {
    #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        free_buf( ( union SIGNAL ** ) &( tmpMessage ) );
        }
    
    for ( message = RTDS_currentContext->writeSaveQueue; message != NULL; )
        {
    #if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        free_buf( ( union SIGNAL ** ) &( tmpMessage ) );
        }
    
    previousProcessInfo = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        /* The queue id is used to find the process id */
        if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )  /* Found ! */
            {
            RTDS_SIMULATOR_TRACE( RTDS_processDied, processInfo, NULL, RTDS_currentContext );
            
            /* Update the process information chained list */
            /* First one in the list */
            if ( previousProcessInfo == NULL )
                {
                RTDS_globalProcessInfo = processInfo->next;
                }
            else
                {
                previousProcessInfo->next = processInfo->next;
                }
            pidToKill = processInfo->myRtosTaskId;
            /* Delete the message queue and free the process info block */
            RTDS_FREE( processInfo );
            break;
            }
        previousProcessInfo = processInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    return pidToKill;
    }

/* **************************************************************** *
 * RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *      - the address of the process current context
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_ProcessKill( RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill;
    
    /* Forget everything about process, delete its message queue and get its PID */
    pidToKill = RTDS_ProcessForget( RTDS_currentContext );
    
    /* Delete the task */
    if ( pidToKill != 0 )
        {
        kill_proc( pidToKill );
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND );
        }
    }

/* **************************************************************** *
 * RTDS_Semaphore_Create
 * **************************************************************** *
 * Create a semaphore
 * **************************************************************** *
 * Parameters:
 *        the initial count of the semaphore
 * Returns:
 *        the semaphore id of the created semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Semaphore_Create( int initialCount, RTDS_SemaphoreType type )
    {
    RTDS_SemaphoreId sem = NULL;
    
    sem = ( RTDS_SemaphoreId )RTDS_MALLOC( sizeof( RTDS_SemType ) );
    
    sem->typeName = type;
    
    switch( type )
        {
        case SEM_COUNTING:
        case SEM_BINARY:
            /* Create the semaphore */
            sem->semId.semaphore = create_sem( initialCount );
            break;
        case SEM_MUTEX:
            /* Create the mutex */
            sem->semId.mutex = ( MUTEX * )RTDS_MALLOC( sizeof( MUTEX ) );
            ose_mutex_init( sem->semId.mutex, OSE_MUTEX_RECURSIVE );
            break;
        default :
            RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEM_TYPE );
            break;
        }
    
    /* Return the id of the semaphore */
    return sem;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdGive
 * **************************************************************** *
 * Attempt to give a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *      - the id of the semaphore
 * Returns:
 *      - nothing
 * **************************************************************** */
void RTDS_SemaphoreIdGive( RTDS_SemaphoreId semaphoreId )
    {
    switch( semaphoreId->typeName )
        {
        case SEM_COUNTING :
        case SEM_BINARY :
            signal_sem( semaphoreId->semId.semaphore );
            break;
        case SEM_MUTEX :
            ose_mutex_unlock( semaphoreId->semId.mutex );
            break;
        default :
            RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEM_TYPE );
            break;
        }
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id and trace it
 * **************************************************************** *
 * Parameters:
 *      - the id of the semaphore
 *      - timeout value
  *      - RTDS_currentContext currentContext of the task
 * Returns:
 *      - status
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_SemaphoreIdTake( RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout timeOut, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt, semaphoreId, timeOut, RTDS_currentContext );
    
    RTDS_SemaphoreTake( semaphoreId, timeOut );
    
    RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded, semaphoreId, ( long )RTDS_GetSemaphoreValue( semaphoreId ), RTDS_currentContext );
    
    return RTDS_OK;
    }

/* **************************************************************** *
 * RTDS_SemaphoreTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *      - the id of the semaphore
 *      - timeout value
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_SemaphoreTake( RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout timeOut )
    {
    /* wait_sem allways blocking. No time out option possible. */
    /* System error if not the case */
    if ( timeOut != RTDS_SEMAPHORE_TIME_OUT_FOREVER )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEMAPHORE_TIME_OUT );
        }
        
    switch( semaphoreId->typeName )
        {
        case SEM_COUNTING :
        case SEM_BINARY :
            wait_sem( semaphoreId->semId.semaphore );
            break;
        case SEM_MUTEX :
            ose_mutex_lock( semaphoreId->semId.mutex );
            break;
        default :
            RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEM_TYPE );
            break;
        }
    }

/* **************************************************************** *
 * RTDS_GetSemaphoreValue
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *      - the id of the semaphore
 * Returns:
 *      - status
 * **************************************************************** */
OSSEMVAL RTDS_GetSemaphoreValue( RTDS_SemaphoreId semaphoreId )
    {
    OSSEMVAL value;
    
    switch( semaphoreId->typeName )
        {
        case SEM_COUNTING :
        case SEM_BINARY :
            value = get_sem( semaphoreId->semId.semaphore );
            break;
        case SEM_MUTEX :
            value = 0;
            break;
        default :
            RTDS_SYSTEM_ERROR( RTDS_ERROR_BAD_SEM_TYPE );
            break;
        }
    
    return value;
    }

/* **************************************************************** *
 * RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *      - semaphore number (name)
 *      - id of the semaphore
 *      - RTDS_currentContext currentContext of the task
 * Returns:
 *      - The semaphore id so that the user can use the semaphore id directly
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Sem_Info_Insert( int semaphoreNumber, RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_GlobalSemaphoreInfo * newSemInfo = NULL;
    
    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );
    if ( newSemInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT_MALLOC );
        }
    
    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;
    
    /* Semaphore creation */
    newSemInfo->semaphoreId = semaphoreId;
    if ( newSemInfo->semaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT );
        }
    
    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( semInfo = RTDS_globalSemaphoreInfo; semInfo->next != NULL; semInfo = semInfo->next );
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated, semaphoreId, -1, RTDS_currentContext );
    return semaphoreId;
    }

/* **************************************************************** *
 * RTDS_Sem_Delete
 * **************************************************************** *
 * Kills an RTOS semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *      - the address of the semaphore to find the semaphore info block
 *      - RTDS_currentContext currentContext of the task
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_Sem_Delete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    RTDS_GlobalSemaphoreInfo * semInfoPrev = NULL;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext );
    
    /* Remove the semaphore information from the chained list */
    semInfoPrev = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreId == semaphoreId ) /* semaphore found */
            {
            if ( semInfoPrev == NULL )
                {
                RTDS_globalSemaphoreInfo = semInfo->next;
                }
            else
                {
                semInfoPrev->next = semInfo->next;
                }
            /* The semaphore should be free for that operation */
            switch( semaphoreId->typeName )
                {
                case SEM_COUNTING :
                case SEM_BINARY :
                    kill_sem( semInfo->semaphoreId->semId.semaphore );
                    break;
                case SEM_MUTEX :
                    ose_mutex_destroy( semInfo->semaphoreId->semId.mutex );
                    RTDS_FREE( semInfo->semaphoreId->semId.mutex );
                    break;
                default :
                    break;
                }
            RTDS_FREE( semInfo );
            break;
            }
        semInfoPrev = semInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/* **************************************************************** *
 * RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *      - semaphore number representing its name
 * Returns:
 *      - the id of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_SemaphoreId foundSemaphoreId = NULL;
    RTDS_GlobalSemaphoreInfo * semInfo = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( foundSemaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID );
        }
    return foundSemaphoreId;    /* Might cause an RTOS exception if NULL*/
    }

/* **************************************************************** *
 * RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *      - None
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo * RTDS_currentContext, int RTDS_sdlStatePrev )
    {
    /* Free message buffer if valid */
    if( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }
        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free memory */
        free_buf( ( union SIGNAL ** ) &( RTDS_currentContext->currentMessage ) );
        RTDS_currentContext->currentMessage = NULL;
        } /* End of else if ( RTDS_currentContext->currentMessage != NULL ) */
    
    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        RTDS_MessageHeader * message;
        /* Let's get to the end of the save queue */
        for( message = RTDS_currentContext->writeSaveQueue; message->next != NULL; message = message->next );
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    return;
    }

/* **************************************************************** *
 * RTDS_TimerCleanUp
 *-------------------------------------------------------------------
 * Called when the last received message is a timer to figure out if
 * timer was not cancelled before it was received.
 *
 * PARAMETERS:
 *     currentContext : RTDS_GlobalProcessInfo* context of the current task
 * **************************************************************** */
void RTDS_TimerCleanUp( RTDS_GlobalProcessInfo * currentContext )
    {
    RTDS_TimerState * timer = NULL;
    RTDS_TimerState * prevTimer = NULL;
    
    prevTimer = NULL;
    for ( timer = currentContext->timerList; timer != NULL; timer = timer->next )
      {
      /* If timer found and cancelled */
      if ( timer->timerUniqueId == currentContext->currentMessage->timerUniqueId )
          {
          if ( timer->state == RTDS_TIMER_CANCELLED )
              {
              /* Discard current message */
              RTDS_FREE( currentContext->currentMessage );
              currentContext->currentMessage = NULL;
              }

          /* Remove it from list of timers */
          if ( prevTimer == NULL )
              {
              currentContext->timerList = currentContext->timerList->next;
              }
          else
              {
              prevTimer->next = timer->next;
              }
          RTDS_FREE( timer );
          break;
          }
      prevTimer = timer;
      }
    }

/* **************************************************************** *
 * RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *      - None
 * Returns:
 *      - System tick count value
 * **************************************************************** */
unsigned long RTDS_GetSystemTime( void )
    {
    return get_systime( NULL );
    }

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER ) || defined( RTDS_FORMAT_TRACE )
unsigned char * RTDS_globalMessageUniqueIdPool = NULL;
RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError, NULL, 0 };
char * RTDS_globalPrintableParameters = NULL;
OSTICK RTDS_globalSystemTime;

/* **************************************************************** *
 * RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *      - None
 * Returns:
 *      - the message unique id (minimum is 1)
 * **************************************************************** */
unsigned long RTDS_GetMessageUniqueId( void )
    {
    long uniqueBitId;
    long uniqueByteId;
    unsigned char * index;
    
    index = RTDS_globalMessageUniqueIdPool;
    
    RTDS_CRITICAL_SECTION_START;
    for( uniqueByteId = 0; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE; uniqueByteId++ )
    {
    if ( *index != 0xFF )
        {
        for ( uniqueBitId = 0; uniqueBitId < 8; uniqueBitId++ )
            {
            if ( ( ( 1 << uniqueBitId ) & *index ) == 0 )
                {
                *index = *index | ( 1 << uniqueBitId );
                RTDS_CRITICAL_SECTION_STOP;
                return ( 8 * uniqueByteId + uniqueBitId + 1 );
                }
            }
        }
    index++;
    }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* All bits are set... No more message unique id */
    RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID );
    return 0;
    }

/* **************************************************************** *
 * RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *      - Message unique id
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
    {
    unsigned char * index;
    
    if ( messageUniqueId == 0 ) /* probably a timer */
        {
        return;
        }
    
    messageUniqueId -= 1;
    index = RTDS_globalMessageUniqueIdPool;
    index += ( unsigned char )( messageUniqueId / 8 );
    
    RTDS_CRITICAL_SECTION_START;
    ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
    RTDS_CRITICAL_SECTION_STOP;
    
    }

/* **************************************************************** *
 * RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue without tracing
 * **************************************************************** *
 * Parameters:
 *      - messageNumber representing a message name
 *      - dataLength length of data pointed by pData
 *      - pData pointer on data sent with the message
 *      - receiver message receiver queue address
 *      - sender message sender queue address
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_SimulatorMsgQueueSend
( long messageNumber,
 long dataLength,
 unsigned char * pData,
 RTDS_SdlInstanceId * receiver,
 RTDS_SdlInstanceId * sender,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader * RTDS_messageToSend;
    
    RTDS_messageToSend = ( RTDS_MessageHeader * )alloc( sizeof ( RTDS_MessageHeader ), messageNumber );
    
    RTDS_messageToSend->messageNumber = messageNumber;
    RTDS_messageToSend->timerUniqueId = 0;
    RTDS_messageToSend->sender = sender;
    RTDS_messageToSend->dataLength = dataLength;
    RTDS_messageToSend->pData = pData;
    RTDS_messageToSend->next = NULL;
    RTDS_messageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
    
    RTDS_SimulatorTrace( ( enum RTDS_EventType )RTDS_messageSent, ( void * )RTDS_messageToSend, ( long )receiver, ( RTDS_GlobalProcessInfo * )RTDS_currentContext, RTDS_DTRACE_ACK_NOWAIT );
    send( ( union SIGNAL ** )( &RTDS_messageToSend ), receiver->queueId );
    }

/* **************************************************************** *
 * RTDS_SimulatorTrace
 * **************************************************************** *
 * As its name states... Handles trace with the host depending on
 * the type of link available and options set in the profile.
 * **************************************************************** *
 * Parameters:
 *      - enum RTDS_EventType  event: trace event
 *      - void *eventParameter1: parameter 1 of the trace
 *      - long eventParameter2: parameter 2 of the trace
 *      - RTDS_GlobalProcessInfo *currentContext: calling context
 *      - int waitAck should the trace be acknowledged (1) or not (0)
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_SimulatorTrace( enum RTDS_EventType event, void * eventParameter1, long eventParameter2, RTDS_GlobalProcessInfo * currentContext, int waitAck )
    {
#ifdef RTDS_FORMAT_TRACE
    char * formatedTrace;
    int retValue;
    
    /* Do not take semaphore when no ack is required i.e. when it comes from the debugger */
    /* i.e. executed in the context of the task reading the socket otherwise dead lock ! */
    if( waitAck == RTDS_DTRACE_ACK_WAIT )
        {
        RTDS_CRITICAL_TRACE_SECTION_START;
        }
#endif
    
    RTDS_globalTraceEntry.event = event;
    RTDS_globalTraceEntry.eventParameter1 = ( void * )eventParameter1;
    RTDS_globalTraceEntry.eventParameter2 = ( long )eventParameter2;
    RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )currentContext;
    
#ifdef RTDS_FORMAT_TRACE
    
    #ifdef RTDS_SIMULATOR
    retValue = RTDS_FormatTrace( ( unsigned long )RTDS_GetSystemTime(), RTDS_globalTraceEntry, waitAck, &formatedTrace );
    #else
    retValue = RTDS_FormatTrace( ( unsigned long )RTDS_GetSystemTime(), RTDS_globalTraceEntry, RTDS_DTRACE_ACK_NOWAIT, &formatedTrace );
    #endif
    
    if( ( retValue == RTDS_OK ) && ( formatedTrace != NULL ) )
        {
    #ifdef RTDS_SOCKET_IP_ADDRESS
        RTDS_SOCKET_ACCESS_TAKE;
        RTDS_SendSocket( formatedTrace, strlen( formatedTrace ) );
        RTDS_SOCKET_ACCESS_GIVE;
    #endif
        if( waitAck == RTDS_DTRACE_ACK_WAIT )
            { 
            RTDS_DTRACE_ACKNOWLEDGE_WAIT;
            }
        }
        RTDS_FREE( formatedTrace );
#endif
    
#ifdef RTDS_SIMULATOR
    RTDS_DummyTraceFunction();
#endif
    
#ifdef RTDS_FORMAT_TRACE
    if( waitAck == RTDS_DTRACE_ACK_WAIT )
        {
        RTDS_CRITICAL_TRACE_SECTION_STOP;
        }
#endif
    }

/* **************************************************************** *
 * RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *      - None
 * Returns:
 *      - Nothing
 * **************************************************************** */
void RTDS_DummyTraceFunction( void )
    {
    }

#endif /* defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACE) || defined(RTDS_FORMAT_TRACE) */

