/* PragmaDev RTDS posix integration */
#ifndef _RTDS_OS_H_
#define _RTDS_OS_H_

#include "RTDS_Proc.h"
#include "RTDS_Scheduler.h"


/*
 * GLOBAL VARIABLES:
 * -------------------
 * Pointers to global list of information on processes and semaphores
 */
#ifdef __cplusplus
extern "C" {
#endif

/* Include for RTDS error messages */
#include "RTDS_Error.h"

/* Include for RTDS types */
#include "RTDS_Common.h"


/*
** MESSAGE FUNCTIONS:
** ------------------
** Handle messages and message queues
*/
extern RTDS_QCB * RTDS_MsgQCreate( void );
extern int RTDS_Wait4ExternalMessage( RTDS_MessageHeader *);

/*
** The function RTDS_incomingSdlEvent should be defined by the user to retrieve
** external messages. The 'timeLeft' parameter is the time left before the next
** timer times-out. If a message is actually received, it should be entered into
** the pre-allocated 'message' parameter and the function should return 1. If no
** message is received within the requested delay, the function should return 0.
*/
extern short RTDS_incomingSdlEvent(RTDS_MessageHeader * message, long timeLeft);


/*
** TIMER FUNCTIONS:
** ----------------
** Handle timers
*/
extern long RTDS_GetTimerUniqueId(void);
extern void RTDS_StartTimer(RTDS_GlobalProcessInfo * context, long timerNumber, long timerUniqueId, int delay);
extern void RTDS_StopTimer(RTDS_GlobalProcessInfo * context, long timerNumber);

/*
** PROCESS FUNCTIONS:
** ------------------
** Handle process creation and deletion
*/
extern void RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext );


extern void RTDS_TimerCleanUp(RTDS_GlobalProcessInfo * currentContext);


/*
** SEMAPHORE FUNCTIONS:
** --------------------
** Handle semaphore creation, registration and manipulation
*/
extern void RTDS_SemaphoreRegister(RTDS_Scheduler * parentScheduler, int semaphoreNumber, RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * creatorContext);
extern RTDS_SemaphoreId RTDS_BinarySemaphoreCreate(RTDS_Scheduler * parentScheduler, int initialState);
extern RTDS_SemaphoreId RTDS_CountingSemaphoreCreate(RTDS_Scheduler * parentScheduler, int initialCount);
extern RTDS_SemaphoreId RTDS_MutexSemaphoreCreate(RTDS_Scheduler * parentScheduler);
extern RTDS_SdlInstanceId * RTDS_GetSemaphoreId(int semaphoreNumber);


/*
** MESSAGE HANDLING FUNCTIONS:
** ---------------------------
*/
#ifdef RTDS_SIMULATOR
extern unsigned long    RTDS_GetMessageUniqueId( void );
extern void             RTDS_ReleaseMessageUniqueId( unsigned long );
extern char           * RTDS_stringToMessageData(long messageNumber, long * dataLength, void ** dataPointer, char * dataString);
extern void             RTDS_messageDataToString(char ** buffer, long messageNumber, long dataLength, void * dataPointer, int level);
#endif


/*
** TRACE FUNCTION:
** ---------------
** Only needed when simulating or tracing
*/
#if defined(RTDS_SOCKET_PORT)
extern int globalClientSocketId;
#endif
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
extern void RTDS_SimulatorTrace( enum RTDS_EventType event , void *eventParameter1 , long eventParameter2 , RTDS_GlobalProcessInfo *currentContext , int waitAck );
#endif


/*
** EXTERNAL VARIABLES:
** -------------------
** Variables declared as global to the all the program
*/
extern RTDS_Scheduler           * RTDS_SystemScheduler;
extern RTDS_GlobalProcessInfo   * RTDS_globalProcessInfo;
extern RTDS_GlobalSemaphoreInfo * RTDS_globalSemaphoreInfo;

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER ) || defined( RTDS_FORMAT_TRACE )
    extern RTDS_GlobalTraceInfo RTDS_globalTraceEntry;
    /* Global variable pointing to decoded parmaters */
    extern char *RTDS_globalPrintableParameters;
#endif

#if defined( RTDS_SIMULATOR )
    extern void *RTDS_globalMessageUniqueIdPool;
#endif


/*
** Macros for message header allocation:
** -------------------------------------
** Can be dynamic or static depending on options
*/
/* If there's a configured maximum number of messages */
#ifdef RTDS_MAX_MESSAGES
/* Message allocation is static: use functions simulating a malloc & a free */
extern RTDS_MessageHeader * RTDS_allocate_message_header(void);
extern void RTDS_free_message_header(RTDS_MessageHeader * message_header);
  
#define RTDS_NEW_MESSAGE_HEADER RTDS_allocate_message_header()
#define RTDS_FREE_MESSAGE_HEADER(MESSAGE_HEADER) RTDS_free_message_header(MESSAGE_HEADER)

/* If there is no configured maximum number of messages */
#else

/* If dynamic memory allocation is allowed */
#if defined(RTDS_MALLOC) && defined(RTDS_FREE)
/* Creating a timer dynamically allocates a descriptor */
#define RTDS_NEW_MESSAGE_HEADER (RTDS_MessageHeader *)RTDS_MALLOC(sizeof(RTDS_MessageHeader))
#define RTDS_FREE_MESSAGE_HEADER(MESSAGE_HEADER) RTDS_FREE(MESSAGE_HEADER)
/* If dynamic memory allocation is not allowed, it's an error */
#else
#error ERROR! Dynamic memory allocation is not available (no RTDS_MALLOC macro), but no maximum number of messages defined via the RTDS_MAX_MESSAGES macro!
#endif

#endif


/*
** Macros for process id & descriptor allocation:
** ----------------------------------------------
** Can be dynamic or static depending on options
*/
/* If there's a configured maximum number of instances */
#ifdef RTDS_MAX_INSTANCES
/* Instance allocation is static: define static array and functions simulating a malloc & a free */
extern RTDS_SdlInstanceId * RTDS_allocate_instance_id(void);
extern RTDS_GlobalProcessInfo * RTDS_allocate_instance_info(void);
extern void RTDS_free_instance_id(RTDS_SdlInstanceId * instance_id);
extern void RTDS_free_instance_info(RTDS_GlobalProcessInfo * instance_info);
  
#define RTDS_NEW_INSTANCE_ID RTDS_allocate_instance_id()
#define RTDS_FREE_INSTANCE_ID(INSTANCE_ID) RTDS_free_instance_id(INSTANCE_ID)

#define RTDS_NEW_INSTANCE_INFO RTDS_allocate_instance_info()
#define RTDS_FREE_INSTANCE_INFO(INSTANCE_INFO) RTDS_free_instance_info(INSTANCE_INFO)

/* If there is no configured maximum number of instances */
#else

/* If dynamic memory allocation is allowed */
#if defined(RTDS_MALLOC) && defined(RTDS_FREE)
/* Creating an instance dynamically allocates an id and a descriptor */
#define RTDS_NEW_INSTANCE_ID (RTDS_SdlInstanceId *)RTDS_MALLOC(sizeof(RTDS_SdlInstanceId))
#define RTDS_FREE_INSTANCE_ID(INSTANCE_ID) RTDS_FREE(INSTANCE_ID)

#define RTDS_NEW_INSTANCE_INFO (RTDS_GlobalProcessInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalProcessInfo))
#define RTDS_FREE_INSTANCE_INFO(INSTANCE_INFO) RTDS_FREE(INSTANCE_INFO)

/* If dynamic memory allocation is not allowed, it's an error */
#else
#error ERROR! Dynamic memory allocation is not available (no RTDS_MALLOC macro), but no maximum number of instances defined via the RTDS_MAX_INSTANCES macro!
#endif

#endif


/*
** Macros and function for timer allocation:
** -----------------------------------------
*/
/* If there's a configured maximum number of timers */
#ifdef RTDS_MAX_TIMERS
/* If this number is 0, timers are not used at all */
#if RTDS_MAX_TIMERS != 0
#define RTDS_HAS_TIMERS
/* If defined and not 0, timer allocation is static: use functions simulating a malloc & a free */
extern RTDS_TimerState * RTDS_allocate_timer_state(void);
extern void RTDS_free_timer_state(RTDS_TimerState * timer_state);
  
#define RTDS_NEW_TIMER_STATE RTDS_allocate_timer_state()
#define RTDS_FREE_TIMER_STATE(TIMER_STATE) RTDS_free_timer_state(TIMER_STATE)
#endif
/* If there is no configured maximum number of timers */
#else
/* If dynamic memory allocation is allowed */
#if defined(RTDS_MALLOC) && defined(RTDS_FREE)
#define RTDS_HAS_TIMERS
/* Creating a timer dynamically allocates a descriptor */
#define RTDS_NEW_TIMER_STATE (RTDS_TimerState *)RTDS_MALLOC(sizeof(RTDS_TimerState))
#define RTDS_FREE_TIMER_STATE(TIMER_STATE) RTDS_FREE(TIMER_STATE)
/* If dynamic memory allocation is not allowed, it's an error */
#else
#error ERROR! Dynamic memory allocation is not available (no RTDS_MALLOC macro), but no maximum number of timers defined via the RTDS_MAX_TIMERS macro!
#endif
#endif


#ifdef __cplusplus
}
#endif

#endif /* End of    _RTDS_OS_H_ */
