/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#include <cmath>
#include <cstdlib>

#include "demoddix.h"


static const double PI = 4.0 * atan(1.0);

int     MainWindow::id;
int     MainWindow::width;
int     MainWindow::height;
int     MainWindow::xPos;
int     MainWindow::yPos;

double  MainWindow::zoomFactor  = 1.0;
double  MainWindow::xMove       = 0.0;
double  MainWindow::yMove       = 0.0;
double  MainWindow::xBegin      = 0.0;
double  MainWindow::yBegin      = 0.0;
bool    MainWindow::isMoving    = false;

bool    MainWindow::showInfo    = false;

// Create window
void MainWindow::Create() 
  {
  MainWindow::width = RootWindow::width - (Window::STATE_W + 3 * Window::PADDING);
  MainWindow::height = RootWindow::height - (Window::PROGRESS_H + 3 * Window::PADDING);
  MainWindow::xPos = Window::STATE_W + 2 * Window::PADDING;
  MainWindow::yPos = Window::PROGRESS_H + 2 * Window::PADDING;
  }

// Draw nodes and packets
void MainWindow::Display() 
  {
  // Vieport and crop
  glViewport(MainWindow::xPos, MainWindow::yPos, MainWindow::width, MainWindow::height);
  glScissor(MainWindow::xPos, MainWindow::yPos, MainWindow::width, MainWindow::height);
  glEnable(GL_SCISSOR_TEST);
  glLoadIdentity();
  glLineWidth(1.0f);
  // Draw
  if (Demoddix::displayStatus() != Demoddix::INIT)
    {
    // Modified zoom considering possible window resize
    double xZoom = (zoomFactor * Window::MAIN_W) / MainWindow::width;
    double yZoom = (zoomFactor * Window::MAIN_W) / MainWindow::height;
    // Ratio for mapping pixels to coordinates
    double xRatio = 2.0 / (xZoom * MainWindow::width);
    double yRatio = 2.0 / (yZoom * MainWindow::height);
    // Move and zoom
    glTranslated(MainWindow::xMove, MainWindow::yMove, 0.0);
    glScaled(xZoom, yZoom, 1.0);
    // Draw packets using lines and arrows (dotted line for lost)
    for (std::list<Packet>::iterator i = Demoddix::packetList.begin(); i != Demoddix::packetList.end(); ++i)
      {
      Packet &p = *i;
      // Skip if corresponding message should not be displayed 
      if (!Demoddix::messageList[p.message].show)
        {
        continue;
        }
      // Coordinates and color
      double xSrc = Demoddix::nodeList[p.source].x;
      double ySrc = Demoddix::nodeList[p.source].y;
      double xDst = Demoddix::nodeList[p.destination].x;
      double yDst = Demoddix::nodeList[p.destination].y;
      unsigned int c = Demoddix::messageList[p.message].color;
      // Draw line
      if (p.lost())
        {
        glEnable(GL_LINE_STIPPLE);
        glLineStipple (1, 0x0F0F);
        }
      glBegin(GL_LINES);
        glColor3ub(Window::COLOR[c][0], Window::COLOR[c][1], Window::COLOR[c][2]);
        glVertex2d(xSrc, ySrc);
        glVertex2d(xDst, yDst);
      glEnd();
      if (p.lost()) 
        {
        glDisable(GL_LINE_STIPPLE);
        }
      // Arrow head coordinates
      double lineAngle = atan2(yDst - ySrc, xDst - xSrc);
      double xOffset = xSrc - xDst;
      double yOffset = ySrc - yDst;
      double scale = (Window::POINT / 2) * xRatio / hypot(xOffset, yOffset);
      double x = xOffset * scale + xDst;
      double y = yOffset * scale + yDst;
      double angle = PI / 9.0;
      double topAngle = lineAngle + PI + angle;
      double bottomAngle = lineAngle + PI - angle;
      double xTop = x + cos(topAngle) * Window::POINT * xRatio;
      double yTop = y + sin(topAngle) * Window::POINT * xRatio;
      double xBottom = x + cos(bottomAngle) * Window::POINT * xRatio;
      double yBottom = y + sin(bottomAngle) * Window::POINT * xRatio;
      // Draw arrow head
      glBegin(GL_TRIANGLE_FAN);
        glColor3ub(Window::COLOR[c][0], Window::COLOR[c][1], Window::COLOR[c][2]);
        glVertex2d(x, y);
        glVertex2d(xTop, yTop);
        glVertex2d(xBottom, yBottom);
      glEnd();
      }
    // Draw nodes using points
    glPointSize(Window::POINT);
    for (unsigned long nId = 0; nId < Demoddix::nodeList.size(); ++nId)
      {
      Node &n = Demoddix::nodeList[nId];
      // Node color
      unsigned int c = Demoddix::stateList[n.state].color;
      // Draw node
      glBegin(GL_POINTS);
        glColor3ub(Window::COLOR[c][0], Window::COLOR[c][1], Window::COLOR[c][2]);
        glVertex2f(n.x, n.y);
      glEnd();
      // Draw id
      double x = n.x + 7.0 * xRatio;
      double y = n.y + 7.0 * yRatio;
      glColor3ub(Window::FG_COLOR[0], Window::FG_COLOR[1], Window::FG_COLOR[2]);
      glRasterPos2d(x, y);
      char str[16];
      sprintf(str, "%lu", nId);
      glutBitmapString(Window::FONT, (const unsigned char *) str);
      // Draw info
      if (MainWindow::showInfo) 
        {
        y = n.y - (7.0 + glutBitmapHeight(Window::FONT)) * yRatio;
        glRasterPos2d(x, y);
        glutBitmapString(Window::FONT, (const unsigned char *) n.name.c_str());
        }
      }
    }
  // Draw border
  glLoadIdentity();
  glLineWidth(2.0f);
  glColor3ub(Window::BD_COLOR[0], Window::BD_COLOR[1], Window::BD_COLOR[2]);
  glBegin(GL_LINE_LOOP);
    glVertex2d(-1.0, -1.0); // bottom left
    glVertex2d(1.0, -1.0); // bottom right
    glVertex2d(1.0, 1.0); // top right
    glVertex2d(-1.0, 1.0); // top left
  glEnd();
  // Disable crop
  glDisable(GL_SCISSOR_TEST);
  }

// Handle window resize
void MainWindow::Reshape() 
  {
  MainWindow::width = RootWindow::width - (Window::STATE_W + 3 * Window::PADDING);
  MainWindow::height = RootWindow::height - (Window::PROGRESS_H + 3 * Window::PADDING);
  }

// Handle special key press
void MainWindow::OnSpecialKeyPress(int key, int x, int y) 
  {
  switch (key) 
    {
    // Rewind
    case GLUT_KEY_LEFT:
      Demoddix::Rewind();
      break;
    // Next
    case GLUT_KEY_UP:
      Demoddix::Next();
      break;
    // Forward
    case GLUT_KEY_RIGHT:
      Demoddix::Forward();
      break;
    // Previous
    case GLUT_KEY_DOWN:
      Demoddix::Previous();
      break;
    // Increase step
    case GLUT_KEY_PAGE_UP:
      Demoddix::stepTime += 1000000; // 1 ms
      break;
    // Decrease step
    case GLUT_KEY_PAGE_DOWN:
      if (Demoddix::stepTime > 1000000)
        {
        Demoddix::stepTime -= 1000000; // 1 ms
        }
      break;
    // Reset view
    case GLUT_KEY_HOME:
      MainWindow::zoomFactor = 1.0;
      MainWindow::xMove = 0.0;
      MainWindow::yMove = 0.0;
      break;
    // Nothing to do
    default:
      return;
    }
  // Update view
  glutPostRedisplay();
  }

// Handle key press
void MainWindow::OnKeyPress(unsigned char key, int x, int y)
  {
  switch (key) 
    {
    // Toggle node info display
    case 'i':
    case 'I':
      if (MainWindow::showInfo)
        {
        MainWindow::showInfo = false;
        } 
      else 
        {
        MainWindow::showInfo = true;
        }
      break;
    // Reset
    case 'r':
    case 'R':
      Demoddix::Reset();
      break;
    // Nothing to do
    default:
      return;
    }
  // Update view
  glutPostRedisplay();
  }

// Handle mouse click
void MainWindow::OnMouseClick(int button, int state, int x, int y) 
  {
  // Ctrl + left mouse on a node opens tracer
  if (glutGetModifiers() == GLUT_ACTIVE_CTRL) 
    {
    if (button == GLUT_LEFT_BUTTON && state == GLUT_DOWN) 
      {
      // Update click coordinates based on window position (viewport) 
      x -= MainWindow::xPos;
      y -= Window::PADDING;
      // Modified zoom factor considering possible window resize
      double xZoom = (MainWindow::zoomFactor * Window::MAIN_W) / MainWindow::width;
      double yZoom = (MainWindow::zoomFactor * Window::MAIN_W) / MainWindow::height;
      // Ratio for mapping pixels to coordinates
      double xRatio = 2.0 / (xZoom * MainWindow::width);
      double yRatio = 2.0 / (yZoom * MainWindow::height);
      // Hot zone considering POINT size
      double x_ = (x - MainWindow::width / 2.0) * xRatio - MainWindow::xMove / xZoom;
      double y_ = (MainWindow::height / 2.0 - y) * yRatio - MainWindow::yMove / yZoom;
      double xHot = (Window::POINT / 2.0) * xRatio;
      double yHot = (Window::POINT / 2.0) * yRatio;
      // Check if any node is in the hot zone
      for (unsigned long nId = 0; nId < Demoddix::nodeList.size(); ++nId)
        {
        Node &n = Demoddix::nodeList[nId];
        if ((x_ >= n.x - xHot && x_ <= n.x + xHot) && (y_ >= n.y - yHot && y_ <= n.y + yHot)) 
          {
          Tracer::Launch((unsigned int) nId);
          break;
          }
        }
      }
    }
  // Move by dragging left mouse or, zoom with mouse wheel
  else 
    {
    // Move
    if (button == GLUT_LEFT_BUTTON) 
      {
      if (state == GLUT_DOWN) 
        {
        MainWindow::xBegin = (2.0 * x) / MainWindow::width - 1.0;
        MainWindow::yBegin = 1.0 - (2.0 * y) / MainWindow::height;
        MainWindow::isMoving = true;
        }
      else 
        {
        MainWindow::xBegin = 0.0;
        MainWindow::yBegin = 0.0;
        MainWindow::isMoving = false;
      }
    }
    // Zoom
    else if (!MainWindow::isMoving) 
      {
      if (button == 3 && state == GLUT_DOWN) 
        {
        MainWindow::zoomFactor *= 1.1;
        glutPostRedisplay();
        }
      else if (button == 4 && state == GLUT_DOWN) 
        {
        MainWindow::zoomFactor /= 1.1;
        glutPostRedisplay();
        }
      }
    }
  }

// Handle mouse move
void MainWindow::OnMouseMove(int x, int y) 
  {
  if (MainWindow::isMoving) 
    {
    // Calculate movement based on mouse position 
    double xTemp = (2.0 * x) / MainWindow::width - 1.0;
    double yTemp = 1.0 - (2.0 * y) / MainWindow::height;
    MainWindow::xMove += xTemp - MainWindow::xBegin;
    MainWindow::yMove += yTemp - MainWindow::yBegin;
    MainWindow::xBegin = xTemp;
    MainWindow::yBegin = yTemp;
    glutPostRedisplay();
    }
  }
