#define _GNU_SOURCE
#define DEV_INPUT_EVENT "/dev/input"
#define EVENT_DEV_NAME "event"
#define DEV_FB "/dev"
#define FB_DEV_NAME "fb"

#include"RaspberryLib.h"


/* ******************************************************************** *
 *                          i2c_write
 * ******************************************************************** *
 * Writes one or two bytes on the i2c bus.
 * ******************************************************************** */ 

void i2c_write(int i2c, unsigned char reg1, unsigned char reg2, int size) {
  uint8_t status = 0;
  unsigned char buffer[16];
  
  buffer[0]=reg1;
  if (size!=1)
    buffer[1]=reg2;
    
  status = write( i2c, buffer, size );
  if ( status == -1 ) {
    if (size == 1)
          printf( "Error writing %02x. Return code: %d\n", buffer[0], status );
    else
      printf( "Error writing %02x %02x. Return code: %d\n", buffer[0], buffer[1], status );
    return;
    }
  }


/* ******************************************************************** *
 *                          i2c_read
 * ******************************************************************** *
 * Reads one byte on the i2c bus.
 * ******************************************************************** */ 

unsigned char i2c_read(int i2c, unsigned char reg1 ) {
  uint8_t status = 0;
  unsigned char buffer[16];
  
  i2c_write(i2c, reg1, 0x00, 1);
  
  status = read( i2c, buffer, 1 );
  if ( status == -1 ) {
    printf( "Error reading %02x. Return code: %d\n", buffer[0], status );
    }

  return buffer[0];
  }

    
/* ******************************************************************** *
 *                          initLSM9DS1
 * ******************************************************************** *
 * Initialize the LSM9DS1 inertial module: accelerometer and gyroscope.
 * ******************************************************************** */ 

int initLSM9DS1(void) {
  int i2c, status;
  struct timespec ts;
  unsigned char buf[16];

  /* Open i2c bus */
  i2c = open( "/dev/i2c-1", O_RDWR );
  if ( i2c == -1 ) {
    printf( "open i2c-1" );
    return -1;
    }

  /* Accelerometer Gyroscope Slave Address */
  status = ioctl( i2c, I2C_SLAVE, LSM9DS1_AG_SAD );  // 0x703, 0x6a
  if ( status == -1 ) {
    printf( "Erreur on ioctl. Return code: %d\n", status );
    return -1;
    }
  
  /* Control register 8 */
  i2c_write(i2c, LSM9DS1_AG_CTRL_REG8, LSM9DS1_AG_CTRL_REG8_BOOT_if(1), 2);

  ts.tv_sec = 0;  ts.tv_nsec = 100000000;
  nanosleep( &ts, NULL );
  
  if ( i2c_read(i2c, LSM9DS1_AG_WHO_AM_I) != LSM9DS1_AG_who_am_i ) {
    printf( "Error reading i2c. Expect 0x%x, get 0x%x\n", LSM9DS1_AG_who_am_i, buf[0] );
    return -1;
    }
  
  /* angular rate sensor control reg 1 */
  i2c_write(i2c, 
    LSM9DS1_AG_CTRL_REG1, 
    LSM9DS1_AG_CTRL_REG1_ODR_G_if(3) | LSM9DS1_AG_CTRL_REG1_FS_G_if(1) | LSM9DS1_AG_CTRL_REG1_BW_G_if(1), 
    2);
  
  /* angular rate sensor control reg 3 */
  i2c_write(i2c, LSM9DS1_AG_CTRL_REG3, LSM9DS1_AG_CTRL_REG3_HP_EN_if(1) | LSM9DS1_AG_CTRL_REG3_HPCF_G_if(4), 2);


  /* linear accel sensor control reg 6 */
  i2c_write(i2c, LSM9DS1_AG_CTRL_REG6_XL, 
    LSM9DS1_AG_CTRL_REG6_XL_ODR_XL_if(3) |
           LSM9DS1_AG_CTRL_REG6_XL_FS_XL_if(3) |
           LSM9DS1_AG_CTRL_REG6_XL_BW_XL_if(3), 2);
          

  /* linear accel sensor control reg 7 */
  i2c_write(i2c, LSM9DS1_AG_CTRL_REG7_XL, 0x00, 2);

  return i2c;
  }

    
/* ******************************************************************** *
 *                          is_event_device
 * ******************************************************************** */ 

int is_event_device(const struct dirent *dir) {
	return strncmp(EVENT_DEV_NAME, dir->d_name,
		       strlen(EVENT_DEV_NAME)-1) == 0;
  }


/* ******************************************************************** *
 *                          is_framebuffer_device
 * ******************************************************************** */ 

int is_framebuffer_device(const struct dirent *dir) {
	return strncmp(FB_DEV_NAME, dir->d_name,
		       strlen(FB_DEV_NAME)-1) == 0;
  }


/* ******************************************************************** *
 *                          open_evdev
 * ******************************************************************** *
 * Opens an event device. An event device is a generic input event 
 * interface in the Linux kernel. It generalizes raw input events from 
 * device drivers and makes them available through character devices in 
 * the /dev/input/ directory.
 * ******************************************************************** */ 

int open_evdev(const char *dev_name)
{
	struct dirent **namelist;
	int i, ndev;
	int fd = -1;

	ndev = scandir(DEV_INPUT_EVENT, &namelist, is_event_device, versionsort);
	if (ndev <= 0)
		return ndev;

	for (i = 0; i < ndev; i++)
	{
		char fname[64];
		char name[256];

		snprintf(fname, sizeof(fname),
			 "%s/%s", DEV_INPUT_EVENT, namelist[i]->d_name);
		fd = open(fname, O_RDONLY);
		if (fd < 0)
			continue;
		ioctl(fd, EVIOCGNAME(sizeof(name)), name);
		if (strcmp(dev_name, name) == 0)
			break;
		close(fd);
	}

	for (i = 0; i < ndev; i++)
		free(namelist[i]);

	return fd;
}


/* ******************************************************************** *
 *                          open_fbdev
 * ******************************************************************** *
 * Opens a frame buffer device. The Linux framebuffer (fbdev) is a 
 * graphic hardware-independent abstraction layer to show graphics on a 
 * computer monitor, typically on the console.[1] The word framebuffer 
 * means a part of video memory containing a current video frame, and 
 * the Linux framebuffer means “access method to the framebuffer under 
 * the Linux kernel”, without relying on system-specific libraries such 
 * as SVGALib or another user space software.
 * ******************************************************************** */ 

int open_fbdev(const char *dev_name) {
	struct dirent **namelist;
	int i, ndev;
	int fd = -1;
	struct fb_fix_screeninfo fix_info;

	ndev = scandir(DEV_FB, &namelist, is_framebuffer_device, versionsort);
	if (ndev <= 0)
		return ndev;

	for (i = 0; i < ndev; i++) {
    char fname[64];
		char name[256];

		snprintf(fname, sizeof(fname),
			 "%s/%s", DEV_FB, namelist[i]->d_name);
		fd = open(fname, O_RDWR);
		if (fd < 0)
			continue;
		ioctl(fd, FBIOGET_FSCREENINFO, &fix_info);
		if (strcmp(dev_name, fix_info.id) == 0)
			break;
		close(fd);
		fd = -1;
    }
	   
	for (i = 0; i < ndev; i++)
		free(namelist[i]);

	return fd;
  }


/* ******************************************************************** *
 *                          Convert24RGBTo16RGB
 * ******************************************************************** *
 * Converts a 24 bits (3x8 bits) RGB value to a 16 bits RGB 
 * (6+5+6 bits) to be sent to the LED panel of the Sense Hat.
 * ******************************************************************** */ 

uint16_t Convert24RGBTo16RGB(unsigned int rgb24) {
  unsigned char red8, green8, blue8, red5, green6, blue5;
  uint16_t rgb16;
  
  red8 = (rgb24 & 0x00FF0000) >> 16;
  green8 = (rgb24 & 0x0000FF00) >> 8;
  blue8 = (rgb24 & 0x000000FF);
  
  red5=(red8*31)/255;
  green6=(green8*63)/255;
  blue5=(blue8*31)/255;
  
  rgb16 = (red5<<11) | (green6<<5) | blue5;
  return rgb16;
  }
  
